#!/usr/bin/env bash

if [ -f "$1" ]; then
  VARS_FILE=$1
else
  VARS_FILE=config.yaml
fi

function display_help() {
  if [ "$1" == "-h" ] || [ "$1" == "--help" ] || [ "$2" == "--help" ] || [ "$2" == "--help" ]; then
    echo -e "Sisense $config_sisense_version Installer"
    echo -e "Usage: $(basename "$0") [CONFIG_YAML_PATH] [flags]\n\n"
    echo -e "-h --help             Display this page"
    echo -e "-y --yes              Skip parameters confirmation page"
    exit 0
  fi
}

source installer/common/functions.sh

function validate_storage_type() {
  if $cluster_mode; then
    is_storage_type_filled=$(yq '.storage_type | length > 0' "$VARS_FILE")
    if ! $is_storage_type_filled; then
      log_to_sisense_installer "You must provide valid storage_type value for cluster mode installations, as mentioned in config file."
      ret 2
      handle_exit_status "validate_storage_type"
    fi
  fi
}

function delete_old_k8s_objects() {
  # In the future, if k8s will deprecate other objects,
  # just add it in the list below, seperated by space
  old_objects="PodSecurityPolicy"

  for object in $old_objects; do
    log_to_sisense_installer "Deleting deprecated objects ${object} before upgrade Kuberenets"
    is_object_exist=$(kubectl api-resources | grep ${object} -c)
    if [[ $is_object_exist -gt 0 ]]; then
      kubectl delete ${object} -A --all --force --grace-period=0 || true
    fi
  done
}

function get_debug_mode() {
  if check_condition ${config_debug_mode}; then
    set -x
  fi
}

function run_or_skip_stage() {
  local stage_number=$1
  local stage_dir=$2
  local stage_description=$3

  if ! is_stage_skipped ${stage_number}; then
    # shellcheck source=./installer/stage_dir/run.sh
    source ${INSTALLER_DIR}/${stage_dir}/run.sh
  else
    log_to_sisense_installer "Skipping stage ${stage_number}: ${stage_description}"
  fi
}

display_help "$1" "$2"
log_to_sisense_installer "Preparing System ..."
load_yaml_files

# Parameter run_as_user will hold the default user for execution
if [[ -z "$config_linux_user" ]]; then
  config_linux_user=$config_run_as_user
fi

get_debug_mode

# Pre validation
source ${INSTALLER_DIR}/01_general_preparation/run.sh

if [ "$1" == "-y" ] || [ "$1" == "--yes" ] || [ "$2" == "--yes" ] || [ "$2" == "-y" ]; then
  if is_on_prem && not_uninstall; then
    run_or_skip_stage 2 "02_sisense_validations" "Sisense validations"
    run_or_skip_stage 3 "03_pre_rke_installation" "Pre RKE installation"
    run_or_skip_stage 4 "04_rke_installation" "RKE installation"
  fi
  run_or_skip_stage 5 "05_post_infra" "Post infra installation"
  if not_uninstall; then
    run_or_skip_stage 6 "06_storage_installation" "Storage installation"
    run_or_skip_stage 7 "07_sisense_installation" "Sisense installation"
    run_or_skip_stage 8 "08_post_installation" "Post Sisense installation"
  fi

  if check_condition ${config_uninstall_sisense} || check_condition ${config_uninstall_cluster}; then
    source ${INSTALLER_DIR}/uninstall/run.sh
  fi
else
  while true; do
    echo -e "The following Configuration will be delegated to Sisense $([[ "$config_uninstall_sisense" == true ]] && echo Removal || echo Installation), Please confirm:"
    yq . <"$VARS_FILE"
    read -rp "Do you wish to $([[ "$config_uninstall_sisense" == true ]] && echo uninstall || echo install) Sisense $config_sisense_version (y/n)? " answer
    case $answer in
      [Yy]*)
        if is_on_prem && not_uninstall; then
          if ! check_condition ${config_skip_validations}; then
            run_or_skip_stage 2 "02_sisense_validations" "Sisense validations"
          else
            log_to_sisense_installer "Skipping Sisense validations"
          fi
          run_or_skip_stage 3 "03_pre_rke_installation" "Pre RKE installation"
          run_or_skip_stage 4 "04_rke_installation" "RKE installation"
        fi
        run_or_skip_stage 5 "05_post_infra" "Post infra installation"
        if not_uninstall; then
          run_or_skip_stage 6 "06_storage_installation" "Storage installation"
          run_or_skip_stage 7 "07_sisense_installation" "Sisense installation"
          run_or_skip_stage 8 "08_post_installation" "Post Sisense installation"
        fi

        if check_condition ${config_uninstall_sisense} || check_condition ${config_uninstall_cluster}; then
          source ${INSTALLER_DIR}/uninstall/run.sh
        fi
        break
        ;;
      [Nn]*) exit ;;
      *) echo -e "Please answer yes or no." ;;
    esac
  done
fi
