#!/usr/bin/env bash

function docker_prune() {
  if check_condition ${old_rke}; then
    if check_condition ${remove_user_data}; then  
      run_command "docker system prune --all --force --volumes > /dev/null" "Clearing Docker images, containers and volumes"
    fi

    if check_condition ${uninstall_cluster} && check_condition ${sisense_cluster} && [[ -z "$rwx_sc_name" ]] && [[ -z "$rwo_sc_name" ]]; then
      log "Removing leftover CNI Docker containers"
      docker ps -a | grep -iE 'calico|flannel|canal|rancher' | awk '{print $1}' | xargs --no-run-if-empty docker rm -f
    fi
  fi
}

function uninstall_rke() {
  if is_on_prem && check_condition ${uninstall_cluster} && check_condition ${sisense_cluster} \
  && ! check_condition ${old_rke} && [[ -z "$rwx_sc_name" ]] && [[ -z "$rwo_sc_name" ]]; then
    local uninstall_script=$(command -v rke2-uninstall.sh || echo "")
    if [[ -n ${uninstall_script} ]]; then
      run_command "sudo ${uninstall_script}" "Uninstalling RKE2 on node ${NODE_NAME}"
    else
      log_royal_blue "No RKE2 uninstallation script found in node ${NODE_NAME}"
    fi
  fi
}

function clean_rook_ceph_block_device() {
  if check_condition ${uninstall_cluster} && [[ ${storage_type,,} == "rook-ceph" ]]; then
    local block_device=$(set | grep ^${NODE_VAR_NAME} | grep disk_volume_device= | cut -d= -f2)
    log "Clearing block device ${block_device} on node ${NODE_NAME}"
    run_command "sudo sgdisk --zap-all ${block_device}"
    run_command "sudo wipefs --all --force ${block_device}"
    run_command "sudo dd if=/dev/zero of='${block_device}' bs=1M count=100 oflag=direct,dsync"
    if [[ $(cat /sys/block/`basename ${block_device}`/queue/discard_max_bytes 2> /dev/null) -gt 0 ]]; then
      # Command below most of times doesn't work, but the rest continues ok, so ingnorable...
      run_command "sudo blkdiscard ${block_device} || true"
    fi
    run_command "sudo bash -c \"ls /dev/mapper/ceph-* 2> /dev/null | xargs -I% -- dmsetup remove %\""
    run_command "sudo wipefs --all --force ${block_device}"
    run_command "sudo rm -rf /dev/mapper/ceph-*"
    run_command "sudo rm -rf /var/lib/rook"
  fi
}

function backup_sisense_logs() {
  if check_condition ${remove_user_data} && [[ -d /var/log/sisense ]]; then
    local current_time=$(date +"%Y%m%dT%H%M%S")
    run_command "sudo tar -czf /var/log/sisense-log-${current_time}.tgz --directory=/var/log sisense" "Backing up sisense logs into /var/log/sisense-log-${current_time}.tgz"
    log_green "Backed up node ${NODE_NAME} sisense logs into /var/log/sisense-log-${current_time}.tgz"
  fi
}

function remove_namespace_dirs() {
  if check_condition ${remove_user_data} && check_condition ${uninstall_sisense}; then
    local dirs="/var/log/sisense/${namespace_name} ${sisense_dir}/mongodb ${sisense_dir}/storage ${sisense_dir}/zookeeper ${sisense_dir}/dgraph-io"
    run_command "sudo rm -rf ${dirs}" "Removing namespaced directories"
  fi
}

function remove_cluster_dirs() {
  if check_condition ${remove_user_data} && check_condition ${uninstall_cluster}; then
    local dirs="/var/log/sisense ${sisense_dir}/mongodb ${sisense_dir}/zookeeper ${sisense_dir}/storage ${sisense_dir}/dgraph-io ${sisense_dir}/alertmanager ${sisense_dir}/prometheus ${sisense_dir}/grafana ${sisense_dir}/config"
    run_command "sudo rm -rf ${dirs}" "Removing cluster directories"
  fi
}

function unmount_kubelet_dirs {
  if check_condition ${uninstall_cluster} && check_condition ${sisense_cluster}; then
    log "Unmounting and deleting kubelet dirs"
    local mounted_dirs=$(mount | grep /var/lib/kubelet/ | awk '{print $3}' | tac)
    for mount_dir in ${mounted_dirs}; do
      run_command "sudo umount -f ${mount_dir} && sudo rm -rf ${mount_dir}"
    done
    log_green "Unmounting kubelet dirs is complete on node ${NODE_NAME}."
  fi
}


################
##### MAIN #####
################
set_node_name_and_ip
detect_os
docker_prune
uninstall_rke
clean_rook_ceph_block_device
backup_sisense_logs
remove_namespace_dirs
remove_cluster_dirs
unmount_kubelet_dirs

log_green "Sisense directories removal finished on node ${NODE_NAME}."
