#!/usr/bin/env bash

WORK_DIR=installer/uninstall
TEMP_CONFIG=${WORK_DIR}/temp_config.yaml
REMOTE_VARS=${WORK_DIR}/remote_vars
REMOTE_SCRIPT=${WORK_DIR}/remote.sh
REMOTE_HOSTS=${WORK_DIR}/hosts
ERROR_DIR=${WORK_DIR}/errors
TEMPLATES_DIR=${WORK_DIR}/templates

source installer/common/functions.sh

RKE_CONFIG_DIR=${sisense_dir}/config/rke
RKE_CONFIG_FILE=${RKE_CONFIG_DIR}/cluster.yml

# Unifying all variables in 1 yaml file
UNIFIED_FILE=/tmp/unified_variables.yaml
yq -y -rs 'reduce .[] as $item ({}; . * $item)' ${DEFAULT_FILE} ${INSTALLER_VALUES} ${VARS_FILE} > ${UNIFIED_FILE}
handle_exit_status "Creating unified params file ${UNIFIED_FILE}"

echo -e "cluster_mode: ${cluster_mode}" >> ${UNIFIED_FILE} || handle_exit_status "Adding cluster mode into unified params file ${UNIFIED_FILE}"
if ! grep -q "storage_type:" ${UNIFIED_FILE}; then
  echo -e "storage_type: ''" >> ${UNIFIED_FILE}  || handle_exit_status "Adding empty storage_type into unified params file ${UNIFIED_FILE}"
fi

# Need to export them for the helmfile.yaml.j2 template later
export FILES_DIR=${WORK_DIR}/files
export HELM_VALUES_DIR=${sisense_dir}/config/umbrella-chart
export LOGMON_DIR=${sisense_dir}/config/logging-monitoring

run_command "sudo mkdir -p ${HELM_VALUES_DIR} && sudo chown -R ${config_linux_user:-$(id -u)}:${config_linux_user:-$(id -g)} ${HELM_VALUES_DIR}"
run_command "sudo mkdir -p ${LOGMON_DIR} && sudo chown -R ${config_linux_user:-$(id -u)}:${config_linux_user:-$(id -g)} ${LOGMON_DIR}"

function install_helmfile() {
  if ! command helmfile > /dev/null 2>&1; then
    if ! check_condition ${config_offline_installer}; then
      local HELMFILE_VERSION="0.159.0"
      local HELMFILE_URL="https://github.com/helmfile/helmfile/releases/download/v${HELMFILE_VERSION}/helmfile_${HELMFILE_VERSION}_linux_amd64.tar.gz"
      run_command "curl -sL ${HELMFILE_URL} | tar -xzO helmfile > ./helmfile" "Downloading Helmfile version v${HELMFILE_VERSION}"
      run_command "sudo mv ./helmfile /usr/local/bin/helmfile && sudo chmod +x /usr/local/bin/helmfile" "Moving Helmfile to /usr/local/bin/helmfile"

      if ! command -v git &> /dev/null ; then
        log_to_sisense_installer "Installing git before helmfile init (helmfile needs git)"
        pkg=$(which dnf 2>/dev/null || which yum 2>/dev/null || which apt 2>/dev/null)
        if [[ -z ${pkg} ]]; then
          log_stderr "Error: no package manager found in host." 1
          handle_exit_status "Git installation before helmfile init"
        fi
        run_command "sudo ${pkg} install git -y"
      fi

      run_command "helmfile init --force > /dev/null 2>&1" "Initializing Helmfile"
    else
      log_stderr "Error: Helmfile not installed, and is offline. Make sure to install Helmfile and run command 'helmfile init --force'." 1
      handle_exit_status "Installing Helmfile"
    fi
  fi
}

function delete_hpa() {
  if check_condition ${config_hpa_enabled}; then
    # Still supporting keda, but below line will be removed in the future
    run_command "kubectl delete scaledobjects -n ${config_namespace_name} --all || true" "Deleting Keda ScaledObjects"
    run_command "kubectl delete hpa -n ${config_namespace_name} --all || true" "Deleting HPAs"
  fi
}

function template_helmfile() {
  local template_file=installer/07_sisense_installation/templates/helmfile.yaml.j2
  local helmfile_file=${HELM_VALUES_DIR}/helmfile.yaml
  export HELM_CHARTS_DIR=${WORK_DIR}/files

  log_to_sisense_installer "Generating Helmfile file before uninstallation"
  template_j2_file ${template_file} ${UNIFIED_FILE} ${helmfile_file}
}

function helmfile_destroy() {
  local helmfile_file=${HELM_VALUES_DIR}/helmfile.yaml
  log_to_sisense_installer "Deploying Sisense using helmfile with file ${helmfile_file}"
  run_command "helmfile --file ${helmfile_file} destroy" "Uninstalling all Helm charts using Helmfile file ${helmfile_file}"

  log_to_sisense_installer "Waiting 30 seconds for pods and resources to terminate gracefully"
  sleep 30
}

function remove_fsx_data() {
  if check_condition ${config_remove_user_data} && [[ ${config_storage_type,,} == "fsx" ]]; then
    local template_file=${TEMPLATES_DIR}/remove-fsx-data-job.yaml.j2
    local job_file=${sisense_dir}/${config_namespace_name}-remove-fsx-data-job.yaml

    log_to_sisense_installer "Generating Cluster Autoscaler values file"
    template_j2_file ${template_file} ${UNIFIED_FILE} ${job_file}

    run_command "kubectl apply -f ${job_file}" "Applying job to Remove FSX Data"
    log_royal_blue "Waiting for Remove FSX Data complete"
    if ! are_all_pods_running "${config_namespace_name}" "app.kubernetes.io/name" "remove-fsx-data" 20 "Succeeded"; then
      log_stderr "Error: Remove FSX Data failed to complete. Ignoring..."
    else
      log_green "Remove FSX Data has completed."
    fi
  fi
}

function delete_pods_leftover() {
  if check_condition ${config_uninstall_cluster} && ( check_condition ${config_internal_monitoring} || check_condition ${config_external_monitoring} ); then
    log_to_sisense_installer "Deleting leftover pods and resources from namespace ${config_monitoring_namespace}"
    run_command "kubectl -n ${config_monitoring_namespace} delete all -l 'app.kubernetes.io/name notin(provisioner)' --force --grace-period 0"
  fi

  if check_condition ${config_remove_user_data}; then
    log_to_sisense_installer "Deleting leftover pods and resources from namespace ${config_namespace_name}"
    run_command "kubectl -n ${config_namespace_name} delete all -l 'app.kubernetes.io/name notin(provisioner)' --force --grace-period 0"
  fi
}

function delete_pvcs_leftover() {
  if check_condition ${config_uninstall_cluster} && ( check_condition ${config_internal_monitoring} || check_condition ${config_external_monitoring} ); then
    log_to_sisense_installer "Deleting leftover PVCs and PVs from namespace ${config_monitoring_namespace}"
    pv_list=$(kubectl get pvc -n ${config_monitoring_namespace} | awk 'NR>1 {print $3}' | tr '\n' ' ')

    log_to_sisense_installer "Patch namespace ${config_monitoring_namespace} PVCs before before deleting them"
    run_command "kubectl -n ${config_monitoring_namespace} get pvc --no-headers | awk '{print \$1}' | xargs --no-run-if-empty kubectl -n ${config_monitoring_namespace} patch pvc -p '{\"metadata\":{\"finalizers\":null}}'"

    log_to_sisense_installer "Deleting namespace ${config_monitoring_namespace} PVCs"
    run_command "kubectl delete pvc --force --grace-period=0 -n ${config_monitoring_namespace} --all"

    if [[ -n ${pv_list} ]]; then
      log_to_sisense_installer "Patching namespace ${config_monitoring_namespace} PVs before before deleting them"
      run_command "kubectl patch pv ${pv_list} -p '{\"metadata\":{\"finalizers\":null}}' || true"

      log_to_sisense_installer "Deleting namespace ${config_monitoring_namespace} remaining PVs"
      run_command "kubectl delete pv --force --grace-period=0 ${pv_list} --ignore-not-found"
    fi    
  fi

  if check_condition ${config_remove_user_data}; then
    log_to_sisense_installer "Configured to remove user data."
    log_to_sisense_installer "Deleting leftover PVCs and PVs from namespace ${config_namespace_name}"
    pv_list=$(kubectl get pvc -n ${config_namespace_name} | awk 'NR>1 {print $3}' | tr '\n' ' ')

    log_to_sisense_installer "Patch namespace ${config_namespace_name} PVCs before before deleting them"
    run_command "kubectl -n ${config_namespace_name} get pvc --no-headers | awk '{print \$1}' | xargs --no-run-if-empty kubectl -n ${config_namespace_name} patch pvc -p '{\"metadata\":{\"finalizers\":null}}'"

    log_to_sisense_installer "Deleting namespace ${config_namespace_name} PVCs"
    run_command "kubectl delete pvc --force --grace-period=0 -n ${config_namespace_name} --all"

    if [[ -n ${pv_list} ]]; then
      log_to_sisense_installer "Patching namespace ${config_namespace_name} PVs before before deleting them"
      run_command "kubectl patch pv ${pv_list} -p '{\"metadata\":{\"finalizers\":null}}' || true"

      log_to_sisense_installer "Deleting namespace ${config_namespace_name} remaining PVs"
      run_command "kubectl delete pv --force --grace-period=0 ${pv_list} --ignore-not-found"
    fi    
  fi
}

function tear_down_rook_ceph() {
  if check_condition ${cluster_mode} && check_condition ${config_uninstall_cluster} && [[ ${config_storage_type,,} == "rook-ceph" ]]; then
    log_to_sisense_installer "Tearing down Rook Ceph"
    local rook_ceph_namespace=rook-ceph    

    declare -A rook_resources
    rook_resources["CephFilesystem"]="rwx-fs"
    rook_resources["CephBlockPool"]="replicapool"
    rook_resources["CephCluster"]="rook-ceph"

    # This way it'll delete first the FileSystem, then BlockPool, and then Cluster - in this order as it should be!
    keys=("CephFilesystem" "CephBlockPool" "CephCluster")

    for resource_type in "${keys[@]}"; do
      resource_name="${rook_resources[$resource_type]}"

      is_resource_exist=$(kubectl -n ${rook_ceph_namespace} get ${resource_type} ${resource_name} --no-headers  --ignore-not-found=true)
      if [[ -n ${is_resource_exist} ]]; then
        run_command "kubectl -n ${rook_ceph_namespace} patch ${resource_type} ${resource_name} -p '{\"metadata\":{\"finalizers\": []}}' --type=merge" "Patching ${resource_type} before delete"
        run_command "kubectl -n ${rook_ceph_namespace} delete ${resource_type} ${resource_name} --force --grace-period=0 --ignore-not-found" "Deleting ${resource_type}"
      fi
    done

    local sc_cephfs=rook-cephfs-r3
    local sc_block=rook-ceph-block
    run_command "kubectl delete sc ${sc_cephfs} ${sc_block} --force --grace-period=0 --ignore-not-found" "Deleting Rook Ceph StorageClasses"

    local rook_ceph_release=rook-operator
    run_command "helm delete -n ${rook_ceph_namespace} ${rook_ceph_release} || true" "Deleting Rook Ceph Helm chart"

    log_to_sisense_installer "Force delete Rook Ceph stuck pods"
    run_command "kubectl -n ${rook_ceph_namespace} get pod --no-headers | { grep Terminating || true; } | awk '{print \$1}' | xargs --no-run-if-empty kubectl -n ${rook_ceph_namespace} delete pod --force --grace-period 0 --ignore-not-found"

    log_green "Rook Ceph was removed from Kubernetes cluster."
  fi 
}

function old_rke_down() {
  if is_on_prem && check_condition ${config_uninstall_cluster} && check_condition ${sisense_cluster} && check_condition ${old_rke} \
    && [[ -z "$config_rwx_sc_name" ]] && [[ -z "$config_rwo_sc_name" ]]; then

    if ! check_condition ${config_within_provisioner}; then
      local rke_binary=$(which rke 2>/dev/null)
      if [[ -n ${rke_binary} ]]; then
        set -o pipefail
        log_to_sisense_installer "Destroying old RKE cluster"
        rke remove --force --config ${RKE_CONFIG_FILE} | tee -a sisense-installer.log 2>&1
        handle_exit_status "Destroying old RKE cluster"
        set +o pipefail
        run_command "sudo rm -f ${rke_binary}" "Removing RKE1 binary"
        log_green "Old RKE cluster destroyed successfully."
      else
        log_stderr "ERROR: Cannot find rke binary, therefore cannot destroy old RKE cluster." 1
        handle_exit_status "Old RKE destruction"
      fi
    else
      log_royal_blue "Cannot destroy old RKE cluster since you are running form Provisioner pod."
      log_to_sisense_installer "In order to destroy the old RKE cluster (if you wish to do so), either run the sisense.sh script, or run command below:"
      log_to_sisense_installer "rke remove --force --config ${RKE_CONFIG_FILE}"
    fi
  fi
}

function remove_rke_and_remote_dirs() {
  if is_on_prem && ( check_condition ${config_uninstall_cluster} || check_condition ${config_remove_user_data} ); then
    if ! check_condition ${config_within_provisioner}; then
      run_remote_script "Sisense uninstallation"
      log_green "Remote infrastructure/Sisense directories removed successfully."
    else
      log_royal_blue "Cannot remove Sisense directories or RKE2 on nodes since you are running form Provisioner pod."
      log_to_sisense_installer "In order to remove Sisense directories or RKE2 (if you wish to do so), run the sisense.sh script."
    fi
  fi
}
