#!/bin/bash
ACTION=$1   # options
CONFIGFILE=$2  # sisense configuration file
PARAMS=$3  # parameters for the specified action

sisense_banner() {
  echo " Sisense Offline Environment installation / configuration 1.0"
  echo "-------------------------------------------------------------"
}

function check_condition() {
  if [ "$1" == "no" ] || [ "$1" == "false" ] || [ "$1" = false ] || [ -z "$1" ]; then
  return 1
  elif [ "$1" == "yes" ] || [ "$1" == "true" ] || [ "$1" = true ]; then
  return 0
  fi
}

image_load_tag_push() {
  echo "Processing $1"
  # Import the image
  sudo ctr -n=k8s.io images import "$1"

  # Get the imported image name from the tar filename
  # Format: registry_org_repo_tag.tar -> registry/org/repo:tag
  tar_basename=$(basename "$1" .tar)
  # Replace last underscore with : for tag
  with_tag=${tar_basename%_*}:${tar_basename##*_}

  # Special handling for sysinfo_sender which contains underscore in name
  if [[ "$with_tag" == *sysinfo_sender* ]]; then
    # Temporarily replace sysinfo_sender with placeholder
    temp=${with_tag//sysinfo_sender/sysinfo-sender}
    # Replace remaining underscores with /
    temp=$(echo "$temp" | tr '_' '/')
    # Restore sysinfo_sender
    image=${temp//sysinfo-sender/sysinfo_sender}
  else
    # Replace remaining underscores with /
    image=$(echo "$with_tag" | tr '_' '/')
  fi

  image_name=$image
  image_name=$(echo "$image_name" | awk -F: '{print $1}' | rev | cut -d'/' -f 1 | rev)
  if [[ $image == *"bitnami"* ]]; then image_name="bitnami/$image_name"; fi
  image_tag=$(echo "$image" | awk -F: '{print $2}')
  new_image=$REGISTRY_SERVER/$image_name:$image_tag
  echo "Tagging & Pushing $new_image to Container Registry"
  sudo ctr -n=k8s.io images tag "$image" "$new_image"
  if [[ ${config_docker_registry_protocol:-http} == "http" ]]; then plain_http="--plain-http"; fi
  sudo ctr -n=k8s.io images push ${plain_http} --user "${USERNAME}:${PASSWORD}" "$new_image"

  if [[ $? -ne 0 ]]; then
    echo "[ERROR] failed to push image ${new_image}"
    exit 1

  elif check_condition ${config_delete_local_images} && [[ ! "$image" == *sisense*provisioner:* ]]; then
    echo "Current image upload done. Deleting image $new_image from local registry"
    sudo ctr -n=k8s.io images rm "$image" "$new_image"
    echo "Deleting file $1 to save space on current server"
    rm -f $1
  fi

  echo "========================================"
}

read_config_file() {
  echo "Reading Configuration files"
  source ./installer/parse_yaml.sh
  eval $(parse_yaml "$CONFIGFILE" "config_")
}

function validate_sudo_permissions() {
  CURRENT_USERNAME=$(id -u -n)

  echo -e "Validating Sudo permissions for user $CURRENT_USERNAME ..."
  if sudo -l -U "$CURRENT_USERNAME" | grep -E -q '.*ALL.*ALL'; then
    echo -e "User $CURRENT_USERNAME has sufficient sudo permissions"
  else
    echo -e "User $CURRENT_USERNAME has insufficient sudo permissions"
    exit 2
  fi
}

upload_images() {
  if [[ -n "$config_docker_registry" ]]; then
    REGISTRY_SERVER=$config_docker_registry
  else
    echo "Container Registry was not set in configuration file"
    exit 2
  fi

  if [[ $# -eq 3 ]]; then
    export USERNAME=${PARAMS%% *}  # Extract everything before the first space
    if [[ "$PARAMS" == *" "* ]]; then
      export PASSWORD=${PARAMS#* } # Extract everything after the first space
    else
      export PASSWORD=""           # Set PASSWORD to empty if no space
    fi
  else
    export USERNAME=$3
    export PASSWORD=$4
  fi

  if [[ -z ${USERNAME} ]]; then
    export USERNAME=${config_docker_username}
    if [[ -z ${USERNAME} ]]; then
    echo "[ERROR] Registry username is missing!"
    print_help
    exit 1
    fi
  fi

  if [[ -z ${PASSWORD} ]]; then
    export PASSWORD=${config_docker_password}
    if [[ -z ${PASSWORD} ]]; then
    read -s -p "Enter registry's password: " PASSWORD
    echo
    export PASSWORD
    fi
  fi

  if [ ! -d "./images" ]; then
    echo "Directory ./images does not exist."
    exit 2
  fi

  FILES="./images/*"
  echo "Uploading Images to Container registry"
  for file in $FILES; do
    if [[ "$file" =~ "docker-base-offline" ]]; then
      continue
    fi

    image_load_tag_push "$file"
  done

  RKE_IMAGES="./rke_images/*"
  REGISTRY_SERVER=$config_docker_registry/rancher
  for image in $RKE_IMAGES; do
    image_load_tag_push "$image"
  done
}

run_sisense_container() {
  CONTAINER_UID=$(id -u)
  SISENSE_DIRECTORY="/opt/sisense"
  KUBE_DIRECTORY="$HOME/.kube"
  SSH_DIRECTORY="$HOME/.ssh"
  CONTAINER_USER=$(id -u -n)
  CONTAINER_GID=$(id -g)

  if check_condition "$config_update" && check_condition "$config_uninstall_sisense" && [[ ! -f "$KUBE_DIRECTORY"/config ]]; then
    echo "Kubeconfig does not exist in expected location: $KUBE_DIRECTORY/config"
    exit 1
  else
    mkdir -p "$KUBE_DIRECTORY"
  fi

  for directory in "$SISENSE_DIRECTORY" "$KUBE_DIRECTORY"; do
    if [[ -d "$directory" ]]; then
      sudo chmod -R g+rw $directory
      DIRECTORY_GROUP=$(find $directory -maxdepth 0 -printf '%g')
      sudo usermod -a -G "$DIRECTORY_GROUP" "$CONTAINER_USER"
    fi
  done

  sudo chown "$CONTAINER_UID" "$KUBE_DIRECTORY" "$SISENSE_DIRECTORY" "$PWD" "$SSH_DIRECTORY"

  for directory in "$SISENSE_DIRECTORY" "$KUBE_DIRECTORY" "$PWD" "$SSH_DIRECTORY"; do
    if [[ $(stat -c '%u' "$directory") != $CONTAINER_UID ]]; then
       sudo chown -R "$CONTAINER_UID" "$directory"
    fi
  done

  SISENSE_PACKAGE_NAME=$(basename "$PWD")
  echo "Entering sisense container for sisense install"

  PROVISIONER_FILE_PATH=$(find ./images/ -maxdepth 1 -name "quay.io_sisense_*_provisioner*" -print)
  SISENSE_VERSION=$(grep ^sisense_version: installer/default.yaml | awk '{print $2}')
  PROVISIONER_IMAGE=$(sudo ctr -n=k8s.io images list | grep "quay.io/sisense.*/provisioner" | grep ${SISENSE_VERSION} | awk '{print $1}')
  if [[ -z ${PROVISIONER_IMAGE} ]]; then
    sudo ctr -n=k8s.io images import "$PROVISIONER_FILE_PATH"
    PROVISIONER_IMAGE=$(sudo ctr -n=k8s.io images list | grep "quay.io/sisense.*/provisioner" | grep ${SISENSE_VERSION} | awk '{print $1}')
  fi

  sudo ctr -n=k8s.io run --rm --tty --allow-new-privs --net-host --user $(id -u):$(id -g) \
  --mount type=bind,src=$SISENSE_DIRECTORY,dst=$SISENSE_DIRECTORY,options=rbind:rw \
  --mount type=bind,src=$HOME,dst=$HOME,options=rbind:rw \
  --mount type=bind,src=$(pwd),dst=$(pwd),options=rbind:rw \
  --mount type=bind,src=/etc/passwd,dst=/etc/passwd,options=rbind:rw \
  --mount type=bind,src=/etc/group,dst=/etc/group,options=rbind:rw \
  --mount type=bind,src=/etc/sudoers,dst=/etc/sudoers,options=rbind:rw \
  --mount type=bind,src=/etc/sudoers.d,dst=/etc/sudoers.d,options=rbind:rw \
  "$PROVISIONER_IMAGE" sisense-container /bin/bash -c "cd $(pwd) && ./sisense.sh ${CONFIGFILE} ${PARAMS}"
}

print_help() {
  sisense_banner
  echo "
  Offline installer provides an easy option to perform Sisense installation in environment without internet access

  Download the latest sisense release with offline images provided by your Sisense Customer Support Manager
  - Upload Sisense image to local registry server

    ./offline_installer.sh upload single_config.yaml <Container Registry Username>

  - Run & Login offline installer container
    ./offline_installer.sh run

  Step By Step Example - upload images and run the offline container local ctr with ready kubernetes cluster

  ========================================================
  1. Verify Kubernetes Cluster integrity
  2. Copy from cluster /etc/kubernetes/admin.conf to ./config
  3. Make sure cluster_config.yaml is set to needed value as described in "https://documentation.sisense.com/latest/linux/offline_install.htm#gsc.tab=0"
  4. Run the following commands
    ./offline_installer.sh upload cluster_config.yaml <Container Registry USERNAME>
    ./offline_installer.sh run

"
}

print_mini_help(){
  sisense_banner
  echo "
  Usage:
  - Upload images to private registry
    ./offline_installer.sh upload cluster_config.yaml <Container Registry USERNAME>

  - run Sisense offline container to execute installer
    ./offline_installer.sh run cluster_config.yaml

  for Additional information run: ./offline_installer.sh help
"
}

if [ -z "${ACTION}" ]; then
  print_mini_help
  echo "** No Action specified **"
  echo ""
  exit
else
  validate_sudo_permissions
  read_config_file
fi

case $ACTION in
  help)
    print_help;;
  upload|images)
    upload_images
    ;;
  run)
    run_sisense_container
    ;;
  *)
    echo "Unknown Action specified"
    print_mini_help
    ;;
esac
