#!/usr/bin/env bash

# Functions for remote.sh script

COLOR_OFF='\033[0m'
RED='\033[0;31m'
GREEN='\033[0;32m'
ROYAL_BLUE='\033[1;36m'
sisense_dir=/opt/sisense

cluster_mode=false
if [[ -n ${storage_type} || -n ${rwx_sc_name} ]]; then 
    cluster_mode=true; 
fi

function set_node_name_and_ip() {
  for line in $(set | grep 'k8s_nodes_.*_internal_ip='); do
    var=$(echo ${line} | cut -d= -f1)
    val=$(echo ${line} | cut -d= -f2)
    export INTERNAL_IP=$(hostname -I | grep -o ${val})

    if [[ -n ${INTERNAL_IP} ]]; then
      # Variable can sometimes be "k8s_nodes_1_internal_ip", "k8s_nodes__1__internal_ip", and other combinations, that's why the 2 sed commands...
      export NODE_VAR_NAME=$(echo ${var} | sed 's/__internal_ip//' | sed 's/_internal_ip//')

      # In single node, the varibale would look like this: k8s_nodes__1='node: node1', that's why the sed commands...
      export NODE_NAME=$(set | grep ^${NODE_VAR_NAME}= | cut -d= -f2 | sed 's/node://g' | sed 's/ //g' | sed "s/'//g")

      if [[ -z ${NODE_NAME} ]]; then
        # In multi node cluster, the variable could be "k8s_nodes_1_node", "k8s_nodes_2_node", etc...
        export NODE_NAME=$(set | grep ^${NODE_VAR_NAME}_node= | cut -d= -f2 | sed 's/node://g' | sed 's/ //g' | sed "s/'//g")
      fi
      break
    fi
  done
}

function ret() { return ${1-0}; }

function check_condition() {
  if [ "$1" == "no" ] || [ "$1" == "false" ] || [ "$1" = false ] || [ -z "$1" ]; then
    return 1
  elif [ "$1" == "yes" ] || [ "$1" == "true" ] || [ "$1" = true ]; then
    return 0
  fi
}

function not_uninstall() {
  if ! check_condition ${uninstall_cluster} && ! check_condition ${uninstall_sisense} && ! check_condition ${remove_user_data}; then
    return 0
  else
    return 1
  fi
}

function is_on_prem() {
  if ! check_condition ${is_kubernetes_cloud} && ! check_condition ${is_openshift}; then
    return 0
  else
    return 1
  fi
}

function log() {
  echo -e "[$(date '+%Y-%m-%d %H:%M:%S')] $*"
}

function log_stderr() {
  # Function simply outputs to stderr, so that the pssh/parallel-ssh could capture the error msg into ${ERROR_DIR} (see functions.sh)
  local msg=$1
  local rc=$2
  log "${RED}** ${msg} **${COLOR_OFF}" >&2
  if [[ -n ${rc} ]]; then ret ${rc}; fi
}

function log_green() {
  local msg=$1
  log "${GREEN}${msg}${COLOR_OFF}"
}

function log_royal_blue() {
  local msg=$1
  log "${ROYAL_BLUE}${msg}${COLOR_OFF}"
}

function handle_exit_status() {
  rc=$?
  if [[ $rc != 0 ]]; then log_stderr "Failed $1 at node ${NODE_NAME} (Internal IP: ${INTERNAL_IP})." && exit $rc; fi
}

function run_command() {
  cmd=$1
  msg=$2
  handle_exit_msg="command ${cmd}"
  if [[ -n ${msg} ]]; then log ${msg}; handle_exit_msg="${msg}"; fi
  eval ${cmd}
  handle_exit_status "${handle_exit_msg}"
}

function detect_os() {
  log "Detecting Host OS  ..."
  if [ -f /etc/os-release ] && [ -f /etc/redhat-release ]; then
    # freedesktop.org and systemd
    . /etc/os-release
    OS=$NAME
    VER=$VERSION_ID
    major_version=$(echo "$VER" | awk -F'.' '{print $1}')
  elif type lsb_release; then
    # linuxbase.org
    OS=$(lsb_release -si)
    VER=$(lsb_release -sr)
  elif [ -f /etc/lsb-release ]; then
    # For some versions of Debian/Ubuntu without lsb_release command
    . /etc/lsb-release
    OS=$DISTRIB_ID
    VER=$DISTRIB_RELEASE
  elif [ -f /etc/debian_version ]; then
    # Older Debian/Ubuntu/etc.
    OS=Debian
    VER=$(cat /etc/debian_version)
  elif [[ -f /etc/system-release && $(cat /etc/system-release) == *"Amazon Linux"* ]]; then
    . /etc/os-release
    OS="Amazon Linux"
    VER=$VERSION_ID
  else
    # Fall back to uname, e.g. "Linux <version>", also works for BSD, etc.
    OS=$(uname -s)
    VER=$(uname -r)
  fi
  handle_exit_status "detect_os"
  log "OS: $OS, Version: $VER"
}

function ini_file_set() {
  # Examply usage:
  # Without sudo
  # ini_file_set /etc/dnf/dnf.conf "main" "http_proxy" "http://something" 644
  # With sudo
  # ini_file_set /etc/dnf/dnf.conf "main" "http_proxy" "http://something" "644" "sudo"

  local file="$1"
  local section="$2"
  local option="$3"
  local value="$4"
  local mode="${5:-644}"
  local sudo="$6"
  local log="$7"

  if [[ -n ${log} ]]; then
    log "Modifying conf file ${file}, section [${section}], ${option}=${value} ..."
    if [[ -n ${sudo} ]]; then
      log "With using sudo"
    else
      log "Without using sudo"
    fi
  fi

  if [[ -f ${file} ]] && ${sudo} grep -q "^\s*\[$section\]" "$file"; then
    if ${sudo} grep -q "^\s*$option\s*=" "$file"; then
      ${sudo} sed -i "s|^\s*$option\s*=.*|$option=$value|" "$file"
    else
      ${sudo} sed -i "/^\s*\[$section\]/a $option=$value" "$file"
    fi
    handle_exit_status "modifying conf file ${file}"
  else
    if [[ -n ${sudo} ]]; then
      echo "[$section]" | sudo tee -a "$file" > /dev/null
      handle_exit_status "adding [$section] to file ${file} with sudo"
      echo -e "${option}=${value}" | sudo tee -a "$file" > /dev/null
      handle_exit_status "adding ${option}=${value} to file ${file} with sudo"
    else
      echo "[$section]" >> "$file"
      handle_exit_status "adding $option=$value to file ${file} without sudo"
      echo -e "${option}=${value}" >> "$file"
      handle_exit_status "adding ${option}=${value} to file ${file} without sudo"
    fi
  fi
  run_command "${sudo} chmod ${mode} ${file}"
}

# In order to use below function, you MUST first set a dictionary called specifically "entries"
# and provide the sysctl_file as the only argument.
# Example:
# declare -A entries=(
#  ["net.bridge.bridge-nf-call-ip6tables"]="1"
#  ["net.bridge.bridge-nf-call-iptables"]="1"
#  ["net.ipv4.ip_forward"]="1"
# )
# set_sysctl_entries /etc/sysctl.d/my-file.conf
function set_sysctl_entries() {
  if [ "$#" -ne 1 ]; then
    echo "Usage: set_sysctl_entries <sysctl_file>"
    ret 1
  fi
  local sysctl_file=$1

  for key in "${!entries[@]}"; do
    local value="${entries[$key]}"
    log "Modifying ${key} = ${value} in file ${sysctl_file}"

    if [[ -f ${sysctl_file} ]] && sudo grep -q "^\s*${key}\s*=" "${sysctl_file}"; then
      sudo sed -i "s|^\s*${key}\s*=.*|${key} = ${value}|" "${sysctl_file}" || handle_exit_status "modifying existing sysctl entry"
    else
      echo "${key} = ${value}" | sudo tee -a ${sysctl_file} > /dev/null || handle_exit_status "adding new sysctl entry"
    fi
  done

  if [[ ${sysctl_file} == "/etc/sysctl.conf" ]]; then
    run_command "sudo sysctl -p"
  else
    run_command "sudo sysctl -p ${sysctl_file}"
  fi
}

function validate_group_and_user() {
  local group_name="$1"
  local user_name="$2"
  local group_id="$3"
  local user_id="$4"

  log "Ensuring user ${user_name} exists under group ${group_name}"
  # Check if the group exists
  if ! getent group "$group_name" >/dev/null; then
    if [ -n "$group_id" ]; then
      # Create the group with the specified group ID
      run_command "sudo groupadd -g $group_id $group_name"
    else
      # Create the group without a specific group ID
      run_command "sudo groupadd $group_name"
    fi
    log "Group '$group_name' created."
  fi

  # Check if the user exists
  if ! id "$user_name" >/dev/null 2>&1; then
    if [ -n "$user_id" ]; then
      # Create the user with the specified user ID and add them to the group
      run_command "sudo useradd -u $user_id -g $group_name -m $user_name"
    else
      # Create the user without a specific user ID and add them to the group
      run_command "sudo useradd -g $group_name -m $user_name"
    fi
    log "User '$user_name' created and added to group '$group_name'."
  else
    # Check if the user is already in the group
    if ! groups "$user_name" | grep -q "\b$group_name\b"; then
      # Add the user to the group
      run_command "sudo usermod -a -G $group_name $user_name"
      log "User '$user_name' added to group '$group_name'."
    fi
  fi
}

function set_property_in_file() {
  local property=$1
  local value=$2
  local file=$3
  local sudo=$4

  # Just incase someone will provide sudo="blabla" or "sudo=yes"...
  if [[ -n ${sudo} ]]; then sudo="sudo"; fi

  if ! ${sudo} grep -q "^${property} ${value}$" ${file}; then
    if ${sudo} grep -q "^${property} " ${file}; then
      log "Setting ${property} ${value} in ${file}"
      ${sudo} sed -i "s/${property}.*/${property} ${value}/g" ${file}
      handle_exit_status "Setting ${property} ${value} in ${file}"
    else
      log "adding ${property} ${value} into ${file}"
      echo "${property} ${value}" | ${sudo} tee -a ${file} > /dev/null
      handle_exit_status "adding ${property} ${value} into ${file}"
    fi
  fi
}
