#!/bin/bash

INSTALLER_DIR=installer
COLOR_OFF='\033[0m'
RED='\033[0;31m'
GREEN='\033[0;32m'
ROYAL_BLUE='\033[1;36m'
REMOTE_FUNCTIONS=${INSTALLER_DIR}/common/remote_functions.sh
EXTRA_VALUES_DIR=${INSTALLER_DIR}/extra_values
DEFAULT_FILE=${INSTALLER_DIR}/default.yaml
INSTALLER_VALUES=${EXTRA_VALUES_DIR}/installer/installer-values.yaml
sisense_dir=/opt/sisense

# Need to export since it's being used in some .j2 files
export EXTRA_VALUES_HELM_DIR=$(pwd)/${EXTRA_VALUES_DIR}/helm

# Meaning: if this runs inside provisioner...
if [[ -f /app/configuration/global-values/prov-values.yml ]]; then
  export EXTRA_VALUES_HELM_DIR=/app/configuration/extra-values

  VARS_FILE=/tmp/prov-values.yml
  cp /app/configuration/global-values/prov-values.yml ${VARS_FILE} || exit 1
  # Copied the prov-values to /tmp so we could edit it if needed
  # (Can't edit files which are generated from ConfigMap such as the prov-values.yml)
fi


function ret() { return ${1-0}; }

function check_condition() {
  if [ "$1" == "no" ] || [ "$1" == "false" ] || [ "$1" = false ] || [ -z "$1" ]; then
    return 1
  elif [ "$1" == "yes" ] || [ "$1" == "true" ] || [ "$1" = true ]; then
    return 0
  fi
}

function not_uninstall() {
  ! check_condition ${config_uninstall_cluster} && ! check_condition ${config_uninstall_sisense} && ! check_condition ${config_remove_user_data}
}

function is_on_prem() {
  ! check_condition ${config_is_kubernetes_cloud} && ! check_condition ${config_is_openshift}
}

function need_infra_installation() {
  is_on_prem && ( ( ! check_condition ${config_update} ) || check_condition ${config_update_k8s_version} || check_condition ${config_change_rke_cidr} || check_condition ${config_recover_kubernetes})
}

function is_stage_skipped() {
  [[ ${config_skip_stages} == *"$1"* ]]
}

function handle_exit_status() {
  rc=$?
  if [[ $rc != 0 ]]; then 
    log_stderr "Error occurred during $1 section"
    log_stderr "Exiting Installation ..."
    exit $rc
  fi
}

function log_to_sisense_installer() {
  echo -e "[$(date '+%Y-%m-%d %H:%M:%S')] $*" | tee -a sisense-installer.log 2>&1
}

function log_stderr() {
  # Function simply outputs to stderr, so that the pssh/parallel-ssh could capture the error msg into ${ERROR_DIR} (see functions.sh)
  local msg=$1
  local rc=$2
  log_to_sisense_installer "${RED}** ${msg} **${COLOR_OFF}" >&2
  if [[ -n ${rc} ]]; then ret ${rc}; fi
}

function log_green() {
  local msg=$1
  log_to_sisense_installer "${GREEN}${msg}${COLOR_OFF}"
}

function log_royal_blue() {
  local msg=$1
  log_to_sisense_installer "${ROYAL_BLUE}${msg}${COLOR_OFF}"
}

function run_command() {
  cmd=$1
  msg=$2
  handle_exit_msg="command ${cmd}"
  if [[ -n ${msg} ]]; then log_to_sisense_installer ${msg}; handle_exit_msg="${msg}"; fi
  set -o pipefail
  eval ${cmd} 2>&1 | tee -a sisense-installer.log 2>&1
  rc=$?
  set +o pipefail
  ret $rc
  handle_exit_status "${handle_exit_msg}"
}

function are_all_nodes_ready() {
  set -o pipefail
  kubectl get nodes | tee -a sisense-installer.log 2>&1
  rc=$?
  set +o pipefail
  if [[ $rc -gt 0 ]]; then
    log_stderr "ERROR: Not connected to Kubernetes cluster."
    return $rc
  fi
  not_ready_nodes=$(kubectl get nodes --no-headers | awk '$2 != "Ready" {print $1}')
  if [[ -n "$not_ready_nodes" ]]; then
    log_stderr "ERROR: The following nodes are not in 'Ready' status:\n$not_ready_nodes" 1
  else
    log_to_sisense_installer "All nodes are in 'Ready' status."
    ret 0
  fi
}

get_kubernetes_version() {
  # Get the server version
  export K8S_SERVER_VERSION=$(kubectl version --short | grep 'Server' | awk '{print $3}')

  # Get the client version
  export K8S_CLIENT_VERSION=$(kubectl version --short | grep 'Client' | awk '{print $3}')

  echo "Kubernetes Server Version: $K8S_SERVER_VERSION"
  echo "Kubernetes Client Version: $K8S_CLIENT_VERSION"
}

function source_parse_yaml() {
  #source parse_yaml function
  source ./installer/parse_yaml.sh
}

function load_yaml_values() {
  yaml_file=$1
  if [[ ! -f ${yaml_file} ]]; then
    log_to_sisense_installer "ERROR: file ${yaml_file} not found."
    exit 1
  fi
  eval $(parse_yaml ${yaml_file} "config_") || exit 1
}

function load_yaml_files() {
  load_yaml_values ${DEFAULT_FILE}
  load_yaml_values ${INSTALLER_VALUES}
  load_yaml_values ${VARS_FILE}
}

function generate_remote_vars() {
  local step_name=$1
  log_to_sisense_installer "Generating remote vars before ${step_name} ..."
  yq -y . $VARS_FILE > ${TEMP_CONFIG} || handle_exit_status "indentation fix for ${VARS_FILE}" # just to fix indentation if needed
  parse_yaml ${DEFAULT_FILE} > ${REMOTE_VARS} || handle_exit_status "parse_yaml ${DEFAULT_FILE}"
  parse_yaml ${INSTALLER_VALUES} >> ${REMOTE_VARS} || handle_exit_status "parse_yaml ${INSTALLER_VALUES}"
  parse_yaml ${TEMP_CONFIG} >> ${REMOTE_VARS} || handle_exit_status "parse_yaml temp_config.yaml"
  echo "sisense_cluster=${sisense_cluster}" >> ${REMOTE_VARS}
  local first_node_name=$(yq '.k8s_nodes[0].node' ${TEMP_CONFIG} | sed 's/"//g')
  echo "first_node_name=${first_node_name}" >> ${REMOTE_VARS}
  if check_condition ${old_rke}; then
      echo "old_rke=${old_rke}" >> ${REMOTE_VARS}
  fi
  echo >> ${REMOTE_VARS}
  if check_condition ${config_remote_installation}; then
    grep "external_ip=" ${REMOTE_VARS} | cut -d= -f2 | sed 's/"//g' | sed "s/'//g" > ${REMOTE_HOSTS}
  else
    grep "internal_ip=" ${REMOTE_VARS} | cut -d= -f2 | sed 's/"//g' | sed "s/'//g" > ${REMOTE_HOSTS}
  fi
}

function run_remote_script() {
  local step_name=$1
  local pssh_tool=""
  local stdbuf="stdbuf -o0"
  local timeout_seconds=600
  local ssh_options="-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ServerAliveInterval=120 -o ServerAliveCountMax=30 -T"
  #ssh_options="${ssh_options} -o ControlMaster=auto -o ControlPersist=30m -o ConnectionAttempts=100"

  log_to_sisense_installer "Checking for parallel-ssh tool ..."
  # In Ubuntu it's "parallel-ssh", in CentOS and others it's "pssh"...
  if which parallel-ssh > /dev/null 2>&1; then 
    pssh_tool=parallel-ssh
  elif which pssh > /dev/null 2>&1; then 
    pssh_tool=pssh
  else
    log_stderr "ERROR: No parallel-ssh or pssh tool found! Cannot run ${step_name}." 1
    handle_exit_status "${step_name}"
  fi

  generate_remote_vars "${step_name}"
  log_to_sisense_installer "Running ${step_name} on k8s nodes ..."
  run_command "rm -rf ${ERROR_DIR} && mkdir -p ${ERROR_DIR}"

  # Using "tr -cd '[:print:]\n'" after remote.sh in order to remove non ascii characters from otuput.
  # That's so the Jenkins automations "pytest" could parse the logs output (pytest doesn't handle non ascii chars...)

  set -o pipefail
  if [[ -n ${config_password} ]]; then
    log_to_sisense_installer "Running ${step_name} using provided ssh password ..."
    # Below is for debug
    # log_to_sisense_installer "cat ${REMOTE_VARS} ${REMOTE_FUNCTIONS} ${REMOTE_SCRIPT} | sshpass -p  \"*****\" ${pssh_tool} -h ${REMOTE_HOSTS} -l ${config_linux_user} -x \"${ssh_options}\" --timeout ${timeout_seconds} -e ${ERROR_DIR} -i -A -I"
    cat ${REMOTE_VARS} ${REMOTE_FUNCTIONS} ${REMOTE_SCRIPT} | sshpass -p "$config_password" ${pssh_tool} -h ${REMOTE_HOSTS} -l ${config_linux_user} -x "${ssh_options}" --timeout ${timeout_seconds} -e ${ERROR_DIR} -i -A -I | tr -cd '[:print:]\n' | ${stdbuf} tee -a sisense-installer.log
  else
    log_to_sisense_installer "Running ${step_name} using provided ssh key ..."
    # Below is for debug
    # log_to_sisense_installer "cat ${REMOTE_VARS} ${REMOTE_FUNCTIONS} ${REMOTE_SCRIPT} | ${pssh_tool} -h ${REMOTE_HOSTS} -l ${config_linux_user} -x \"-i ${config_ssh_key} ${ssh_options}\" --timeout ${timeout_seconds} -e ${ERROR_DIR} -i -I"
    cat ${REMOTE_VARS} ${REMOTE_FUNCTIONS} ${REMOTE_SCRIPT} | ${pssh_tool} -h ${REMOTE_HOSTS} -l ${config_linux_user} -x "-i ${config_ssh_key} ${ssh_options}" --timeout ${timeout_seconds} -e ${ERROR_DIR} -i -I | tr -cd '[:print:]\n' | ${stdbuf} tee -a sisense-installer.log
  fi

  if [[ $? -ne 0 ]]; then
    # Second try, just incasae of "client_loop: send disconnect: Broken pipe" error
    log_to_sisense_installer ""
    log_to_sisense_installer "Running ${step_name} second and last try..."
    if [[ -n ${config_password} ]]; then
      log_to_sisense_installer "Running ${step_name} using provided ssh password ..."
      # Below is for debug
      # log_to_sisense_installer "cat ${REMOTE_VARS} ${REMOTE_FUNCTIONS} ${REMOTE_SCRIPT} | sshpass -p  \"*****\" ${pssh_tool} -h ${REMOTE_HOSTS} -l ${config_linux_user} -x \"${ssh_options}\" --timeout ${timeout_seconds} -e ${ERROR_DIR} -i -A -I"
      cat ${REMOTE_VARS} ${REMOTE_FUNCTIONS} ${REMOTE_SCRIPT} | sshpass -p "$config_password" ${pssh_tool} -h ${REMOTE_HOSTS} -l ${config_linux_user} -x "${ssh_options}" --timeout ${timeout_seconds} -e ${ERROR_DIR} -i -A -I | tr -cd '[:print:]\n' | ${stdbuf} tee -a sisense-installer.log
    else
      log_to_sisense_installer "Running ${step_name} using provided ssh key ..."
      # Below is for debug
      # log_to_sisense_installer "cat ${REMOTE_VARS} ${REMOTE_FUNCTIONS} ${REMOTE_SCRIPT} | ${pssh_tool} -h ${REMOTE_HOSTS} -l ${config_linux_user} -x \"-i ${config_ssh_key} ${ssh_options}\" --timeout ${timeout_seconds} -e ${ERROR_DIR} -i -I"
      cat ${REMOTE_VARS} ${REMOTE_FUNCTIONS} ${REMOTE_SCRIPT} | ${pssh_tool} -h ${REMOTE_HOSTS} -l ${config_linux_user} -x "-i ${config_ssh_key} ${ssh_options}" --timeout ${timeout_seconds} -e ${ERROR_DIR} -i -I | tr -cd '[:print:]\n' | ${stdbuf} tee -a sisense-installer.log
    fi

    if [[ $? -ne 0 ]]; then
      # This way, if ${step_name} fails at one of the nodes, then it'll print the error messages at the end of the log
      # This will save customers/support time that could've been wasted on searching for the error deep in the log
      log_to_sisense_installer ""
      log_stderr "${step_name} failed."
      for file in ${ERROR_DIR}/*; do
        log_to_sisense_installer "Errors from host $(basename ${file}):"
        cat ${file} | tr -cd '[:print:]\n' | tee -a sisense-installer.log
      done
      ret 9
    fi
  fi
  handle_exit_status "${step_name}"
  set +o pipefail
}

function set_property_in_file() {
  local property=$1
  local value=$2
  local file=$3
  local sudo=$4

  # Just incase someone will provide sudo="blabla" or "sudo=yes"...
  if [[ -n ${sudo} ]]; then sudo="sudo"; fi

  if ! ${sudo} grep -q "^${property} ${value}$" ${file}; then
    if ${sudo} grep -q "^${property} " ${file}; then
      log_to_sisense_installer "Setting ${property} ${value} in ${file}"
      ${sudo} sed -i "s/${property}.*/${property} ${value}/g" ${file}
      handle_exit_status "Setting ${property} ${value} in ${file}"
    else
      log_to_sisense_installer "adding ${property} ${value} into ${file}"
      echo "${property} ${value}" | ${sudo} tee -a ${file} > /dev/null
      handle_exit_status "adding ${property} ${value} into ${file}"
    fi
  fi
}


function version_gt() { test "$(echo "$@" | tr " " "\n" | sort -V | head -n 1)" != "$1"; }
function version_le() { test "$(echo "$@" | tr " " "\n" | sort -V | head -n 1)" == "$1"; }
function version_lt() { test "$(echo "$@" | tr " " "\n" | sort -rV | head -n 1)" != "$1"; }
function version_ge() { test "$(echo "$@" | tr " " "\n" | sort -rV | head -n 1)" == "$1"; }

function get_pods_status() {
  local namespace=$1
  local selector_key=$2
  local selector_value=$3

  kubectl get po -n "${namespace}" --selector="${selector_key}=${selector_value}" --sort-by='.metadata.creationTimestamp' -o jsonpath='{.items[-1].status.phase}' 2> /dev/null
}

function are_all_pods_running() {
  local namespace=$1
  local selector_key=$2
  local selector_value=$3
  local retries=$4
  local desired_state=$5

  for (( attempt=1; attempt<=$retries; attempt++ )); do
    log_royal_blue "Waiting for ${selector_value} pods to be ${desired_state,,} ... Attempt ${attempt}/${retries}"
    pod_statuses=($(get_pods_status "${namespace}" "${selector_key}" "${selector_value}"))
    all_running=true
    if [[ ${#pod_statuses[@]} -eq 0 ]]; then
      all_running=false
    fi
    for status in "${pod_statuses[@]}"; do
      if [[ "${status}" != "${desired_state}" ]]; then
        all_running=false
        break
      fi
    done

    if $all_running; then
      log_green "All pods of ${selector_value} service are ${desired_state,,}"
      return 0
    fi
    sleep 5
  done

  log_to_sisense_installer "Pods of ${selector_value} service did not become ${desired_state,,} after ${retries} retries"
  return 1
}

function get_label_value() {
  echo $(kubectl get no -L$1 | awk 'NR>1 {print $6}' | awk 'NR' | sort | uniq)
}

function template_j2_file() {
  # Note: if you want to use also bash variables, you must export them first!
  # Example: assume you have {{ my_var }} and it's a bash variable.
  # Then do "export my_var=my_value" before calling this function
  
  template_file=$1
  values_file=$2
  actual_file=$3
  
  log_to_sisense_installer "Evaluting template file ${template_file} into ${actual_file}"
  jinjanate -e "" -f yaml ${template_file} ${values_file} > ${actual_file}
  handle_exit_status "Evaluting template file ${template_file} into ${actual_file}"
}

function helm_mapkubeapis() {
  local release=$1
  local namespace=$2

  if [[ -z ${release} || -z ${namespace} ]]; then
    log_stderr "Error: called function 'helm_mapkubeapis' with release or namespace are undefined!"
    log_stderr "Usage: helm_mapkubeapis <release-name> <namespace>" 1
    handle_exit_status "function 'helm_mapkubeapis'"
  fi

  if helm ls -n ${namespace} | awk '{print $1}' | grep -q "^${release}$"; then
    log_royal_blue "Updating Kubernetes API versions on Helm release '${release}' in namespace '${namespace}'"
    run_command "helm mapkubeapis -n ${namespace} ${release}"
  fi
}

function assign_to_helm_release() {
  local resource=$1
  local resource_name=$2
  local namespace=$3
  local release=$4

  is_resource_exist=$(kubectl -n ${namespace} get ${resource} ${resource_name} --no-headers --ignore-not-found)
  if [[ -n ${is_resource_exist} ]]; then
    log_royal_blue "Labeling ${resource} ${resource_name} to be managed by helm"
    run_command "kubectl -n ${namespace} label ${resource} ${resource_name} app.kubernetes.io/managed-by=Helm --overwrite=true"

    log_royal_blue "Annotating ${resource} ${resource_name} to be managed by Helm release ${release} in namespace ${namespace}"
    kubectl -n ${namespace} annotate ${resource} ${resource_name} \
      meta.helm.sh/release-namespace=${namespace} \
      meta.helm.sh/release-name=${release} \
      --overwrite=true
    handle_exit_status "Annotating ${resource} ${resource_name} to be managed by Helm release ${release} in namespace ${namespace}"
  fi
}

function generate_overrided_params() {
  TEMP_DIR=/tmp/sisense
  OVERRIDE_FILE=${TEMP_DIR}/overrided_params.yaml

  log_to_sisense_installer "Generating overriding params file ${OVERRIDE_FILE}"
  run_command "touch ${OVERRIDE_FILE}"
  run_command "cat /dev/null >${OVERRIDE_FILE}"
  log_to_sisense_installer "INFO: Getting Kubernetes Cloud Provider and location ..."

  ## Get openshift
  IS_OPENSHIFT=$(get_label_value "node.openshift.io/os_id")

  ## Get cloud provider
  CLOUD_PROVIDER=$(kubectl get no -o jsonpath='{.items[0].spec.providerID}')

  ## Get cloud region
  CLOUD_REGION=$(get_label_value "failure-domain.beta.kubernetes.io/region")
  if [[ ${CLOUD_REGION} == "" ]]; then
    CLOUD_REGION=$(get_label_value "topology.kubernetes.io/region")
  fi

  if [[ ${IS_OPENSHIFT} != "" ]]; then
    {
      echo -e "is_openshift: true"
      echo -e "is_kubernetes_cloud: false"
    } >>${OVERRIDE_FILE}
  elif [[ ${CLOUD_PROVIDER} == "" || ${CLOUD_PROVIDER} == k3s* || ${CLOUD_PROVIDER} == rke2* ]]; then ## might need to cherry pick this to old versions...
    {
      echo -e "is_openshift: false"
      echo -e "is_kubernetes_cloud: false"
    } >>${OVERRIDE_FILE}
  elif [[ ${CLOUD_PROVIDER} != "" ]]; then
    {
      echo -e "is_openshift: false"
      echo -e "is_kubernetes_cloud: true"
    } >>${OVERRIDE_FILE}
  fi

  if [[ ${CLOUD_PROVIDER} == *"aws"* ]]; then
    EKS_CLUSTER_NAME=$(get_label_value "alpha.eksctl.io/cluster-name")

    if [[ ${EKS_CLUSTER_NAME} == "" ]]; then
      AWS_METADATA_URL=169.254.169.254
      INSTANCE_ID=$(curl -s ${AWS_METADATA_URL}/latest/meta-data/instance-id -m 10)
      if [[ -z ${INSTANCE_ID} ]]; then
        TOKEN=$(curl -sX PUT "http://${AWS_METADATA_URL}/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 21600")
        INSTANCE_ID=$(curl -sH "X-aws-ec2-metadata-token: $TOKEN" "http://${AWS_METADATA_URL}/latest/meta-data/instance-id" -m 10)
      fi

      EKS_CLUSTER_NAME=$(aws ec2 describe-tags \
        --region ${CLOUD_REGION} \
        --filters "Name=resource-id,Values=${INSTANCE_ID}" \
        --query 'Tags[?starts_with(Key, `kubernetes.io/cluster/`) == `true`].Key' --output text | cut -f3 -d'/')
    fi

    {
      echo -e "kubernetes_cloud_provider: aws"
      echo -e "kubernetes_cluster_name: ${EKS_CLUSTER_NAME}"
      echo -e "kubernetes_cluster_location: ${CLOUD_REGION}"
    } >>${OVERRIDE_FILE}

  elif [[ ${CLOUD_PROVIDER} == *"azure"* ]]; then
    AKS_CLUSTER_NAME=$(get_label_value "kubernetes.azure.com/cluster")

    {
      echo -e "kubernetes_cloud_provider: azure"
      echo -e "kubernetes_cluster_name: ${AKS_CLUSTER_NAME}"
      echo -e "kubernetes_cluster_location: ${CLOUD_REGION}"
    } >>${OVERRIDE_FILE}

  elif [[ ${CLOUD_PROVIDER} == *"gce"* ]]; then
    GKE_CLUSTER_NAME=$(curl -H "Metadata-Flavor: Google" http://metadata.google.internal/computeMetadata/v1/instance/attributes/cluster-name)
    {
      echo -e "kubernetes_cloud_provider: gke"
      echo -e "kubernetes_cluster_name: ${GKE_CLUSTER_NAME}"
      echo -e "kubernetes_cluster_location: ${CLOUD_REGION}"
    } >>${OVERRIDE_FILE}

  fi

  log_to_sisense_installer "INFO: Configuration Completed"
}

function generate_overrided_params_wrapper() {
  TEMP_DIR=/tmp/sisense
  OVERRIDE_FILE=${TEMP_DIR}/overrided_params.yaml
  run_command "mkdir -p ${TEMP_DIR}"
  cat /dev/null > ${OVERRIDE_FILE}

  # Incase "cluster_visibility" was not provided, then default value is "true"
  config_cluster_visibility=${config_cluster_visibility:-true}

  if check_condition ${config_cluster_visibility}; then
    if ! check_condition ${config_is_openshift}; then
      if ! check_condition ${config_kubernetes_cloud_provider} || ! check_condition ${config_kubernetes_cluster_name} || ! check_condition ${config_kubernetes_cluster_location}; then
        generate_overrided_params
      fi
    fi
  else
    echo -e "cluster_visibility: false" >> ${OVERRIDE_FILE}
  fi
  load_yaml_values ${OVERRIDE_FILE}

  # Unifying all variables in 1 yaml file
  UNIFIED_FILE=/tmp/unified_variables.yaml
  yq -y -rs 'reduce .[] as $item ({}; . * $item)' ${DEFAULT_FILE} ${INSTALLER_VALUES} ${VARS_FILE} ${OVERRIDE_FILE} > ${UNIFIED_FILE}
  handle_exit_status "Creating unified params file ${UNIFIED_FILE}"

  if ! grep -q "cluster_mode:" ${UNIFIED_FILE}; then
    echo -e "cluster_mode: ${cluster_mode}" >> ${UNIFIED_FILE} || handle_exit_status "Adding cluster mode into unified params file ${UNIFIED_FILE}"
  fi
  if ! grep -q "storage_type:" ${UNIFIED_FILE}; then
    echo -e "storage_type: ''" >> ${UNIFIED_FILE}  || handle_exit_status "Adding empty storage_type into unified params file ${UNIFIED_FILE}"
  fi
}

source_parse_yaml
load_yaml_files

cluster_mode=false
if [[ -n ${config_storage_type} || -n ${config_rwx_sc_name} ]]; then
    cluster_mode=true
fi
