#!/usr/bin/env bash

source installer/common/functions.sh

function validator_job() {
  if ! check_condition "$config_uninstall_sisense" && ! check_condition "$config_uninstall_cluster"; then
    local validator_job=0
    local validator_wait=1200s
    local validator_label="-l app.kubernetes.io/name=validator"
    local validator_params="-n ${config_namespace_name} ${validator_label}"
    local wait_params="${validator_params} --timeout=${validator_wait}"

    trap '(( validator_job |= $? ))' ERR
    log_to_sisense_installer "Setting up validator execution..."
    kubectl wait --for=condition=PodScheduled pod ${wait_params} >/dev/null 2>&1

    # Just for an edge case, if the validator will be "Succeeded" super fast, so it won't be in "ready" at this point...
    local status=$(kubectl get pod ${validator_params} --sort-by='.metadata.creationTimestamp' -o jsonpath='{.items[-1].status.phase}')
    if [[ ${status} != "Succeeded" ]]; then
      kubectl wait --for=condition=ready pod ${wait_params} >/dev/null 2>&1
    fi

    trap - ERR
    if [[ $validator_job != 0 ]]; then
      log_stderr "ERROR: Failed to initialize or schedule Sisense validator"
      ret 2
      handle_exit_status "validator job"
    else
      kubectl logs -f ${validator_params} --tail=-1 | tee -a sisense-installer.log 2>&1
      handle_exit_status "tailing validators log"

      log_to_sisense_installer "Waiting for validator results ..."
      latest_job=$(kubectl get job ${validator_params} --sort-by='.metadata.creationTimestamp' -o jsonpath='{.items[-1].metadata.name}')

      # If job failed, then the "kubectl wait" will be code 0, 
      # so in that case it'll do "log_stderr" with code 1, and then handle_exit_status (and exit 1)
      kubectl wait --for=condition=Failed job -n ${config_namespace_name} ${latest_job} --timeout=0 > /dev/null 2>&1 && \
      { log_stderr "ERROR: Sisense validator has failed" 1; handle_exit_status "validator job"; }

      kubectl wait --for=condition=Complete job -n ${config_namespace_name} ${latest_job} --timeout=${validator_wait} > /dev/null 2>&1 || \
      { log_stderr "ERROR: Sisense validator has failed" 1; handle_exit_status "validator job"; }

      log_green "Validator completed."
      log_green "Sisense installation completed."
    fi    
  fi
}
