#!/usr/bin/env bash

WORK_DIR=installer/07_sisense_installation
TEMP_CONFIG=${WORK_DIR}/temp_config.yaml
REMOTE_VARS=${WORK_DIR}/remote_vars
REMOTE_SCRIPT=${WORK_DIR}/remote.sh
REMOTE_HOSTS=${WORK_DIR}/hosts
ERROR_DIR=${WORK_DIR}/errors
TEMPLATES_DIR=${WORK_DIR}/templates

# Needed to export for using in helmfile.yaml.j2
export FILES_DIR=${WORK_DIR}/files

source installer/common/functions.sh

export HELM_VALUES_DIR=${sisense_dir}/config/umbrella-chart
export LOGMON_DIR=${sisense_dir}/config/logging-monitoring
export kubernetes_version=$(kubectl version --short -ojson 2>/dev/null | jq '.serverVersion.gitVersion' | sed 's/"//g')
handle_exit_status "Getting kubectl version"
export k8s_node_1=$(kubectl get no -o jsonpath='{.items[0].metadata.name}')
handle_exit_status "Getting Kubernetes first node name"
export k8s_node1_ip_default=$(kubectl get no -o jsonpath='{.items[0].status.addresses[?(@.type=="InternalIP")].address}')
handle_exit_status "Getting Kubernetes first node internal IP address"

run_command "sudo mkdir -p ${HELM_VALUES_DIR} && sudo chown -R ${config_linux_user:-$(id -u)}:${config_linux_user:-$(id -g)} ${HELM_VALUES_DIR}"
run_command "sudo mkdir -p ${LOGMON_DIR} && sudo chown -R ${config_linux_user:-$(id -u)}:${config_linux_user:-$(id -g)} ${LOGMON_DIR}"

function configure_auto_labeling() {
  # Below happens mostly only in Provisioner, if user configured to auto label node via our logic
  # (Our logic: all k8s nodes do all roles)
  if check_condition ${config_label_all_nodes}; then
    log_royal_blue "Auto node labeling configured"
    log_royal_blue "Overwriting k8s_node in the values file, with all current Kubernetes nodes"
    log_royal_blue "All current Kubernetes nodes will be used for application, query, and build"

    node_count=$(kubectl get nodes --no-headers |wc -l)
    if [[ ${node_count} -eq 1 ]]; then
      node_name=$(kubectl get nodes -ojsonpath='{.items[0].metadata.name}')
      run_command "yq -y '.k8s_nodes = [{\"node\": \"${node_name}\", \"roles\": \"application, query, build\"}]' ${VARS_FILE} > /tmp/temp.yml"
    else
      node_names=$(kubectl get nodes -ojsonpath='{.items[*].metadata.name}')
      elements=""
      first_iteration=true

      for node_name in ${node_names}; do
        if check_condition ${first_iteration}; then
          elements="{\"node\": \"${node_name}\", \"roles\": \"application, query, build\"}"
          first_iteration=false
        else
          elements="${elements}, {\"node\": \"${node_name}\", \"roles\": \"application, query, build\"}"
        fi
      done

      run_command "yq -y '.k8s_nodes = [${elements}]' ${VARS_FILE} > /tmp/temp.yml"
    fi
    run_command "cp /tmp/temp.yml ${VARS_FILE}"
    load_yaml_files
  fi
}

function label_nodes() {
  if ! check_condition ${config_skip_labeling} && ! check_condition ${config_is_openshift} && check_condition ${config_cluster_visibility}; then
    if ! check_condition ${cluster_mode}; then
      local node_name=$(yq '.k8s_nodes[0].node' ${VARS_FILE} | sed 's/"//g')
      run_command "kubectl label node ${node_name} node=single --overwrite=true" "Adding single label to node ${node_name}"
    else
      local node_count=$(yq '.k8s_nodes | length' ${VARS_FILE})
      local node_count_last=$(expr $node_count - 1)

      if [[ ${config_kubernetes_cloud_provider,,} == "azure" ]]; then
        kubectl label nodes --all node-${config_namespace_name}-Application- node-${config_namespace_name}-Build- node-${config_namespace_name}-Query-
        rc=$?
        if [[ ${rc} -ne 0 ]]; then
          log_stderr "Relabelling k8s nodes is unsupported on Azure AKS when labels are managed via Azure AKS itself."
          log_royal_blue "You should either remove Sisense node labels from the AKS via azure UI or az command,"
          log_royal_blue "or you can simply set 'skip_labeling: true' in the installer extra values to skip this message."
          log_to_sisense_installer
          log_stderr "Currenetly Sisense will work with current existing node labels"
          log_to_sisense_installer "Continuing with the rest of the installation"
          return
        fi
      fi

      run_command "kubectl label nodes --all node-${config_namespace_name}-Application- node-${config_namespace_name}-Build- node-${config_namespace_name}-Query-"
      for i in $(seq 0 $node_count_last); do
        local node_name=$(yq ".k8s_nodes[${i}].node" ${VARS_FILE} | sed 's/"//g')
        local node_roles=$(yq ".k8s_nodes[${i}].roles" ${VARS_FILE} | sed 's/"//g')
        
        if [[ ${node_roles,,} == *"application"* ]]; then
          run_command "kubectl label node ${node_name} node-${config_namespace_name}-Application=true --overwrite=true" "Adding application label to node ${node_name}"
        fi
        if [[ ${node_roles,,} == *"query"* ]]; then
          run_command "kubectl label node ${node_name} node-${config_namespace_name}-Query=true --overwrite=true" "Adding query label to node ${node_name}"
        fi
        if [[ ${node_roles,,} == *"build"* ]]; then
          run_command "kubectl label node ${node_name} node-${config_namespace_name}-Build=true --overwrite=true" "Adding build label to node ${node_name}"
        fi
        run_command "kubectl taint nodes ${node_name} node-role.kubernetes.io/master:NoSchedule- >/dev/null 2>&1 || true" "Remove master NoSchedule taint fron node ${node_name} (Ignoring not found...)"
      done
    fi
  elif check_condition ${cluster_mode} && check_condition ${config_cluster_visibility}; then
    run_command "kubectl taint nodes --all node-role.kubernetes.io/master:NoSchedule- >/dev/null 2>&1 || true" "Remove master NoSchedule taint from all nodes (Ignoring not found...)"
  fi
}

function validate_sc_existing() {
  if [[ -n ${config_rwx_sc_name} || -n ${config_rwo_sc_name} ]]; then
    # Need to check if also to add contidion 
    # if [[ -n ${config_storage_type} && ( -n ${config_rwx_sc_name} || -n ${config_rwo_sc_name} ) ]]
    # To prevent clash between storage_type (not "nfs-server") and RWX/RWO - maning: dude, choose one of them, not both...

    if [[ ( -z ${config_storage_type} && ( -z ${config_rwx_sc_name} || -z ${config_rwo_sc_name} ) ) || ( "$config_storage_type" == "nfs-server" && -z ${config_rwo_sc_name} ) ]]; then
      log_stderr "Incorrect input for storage_type (${config_storage_type}), rwx_sc_name (${config_rwx_sc_name}), and rwo_sc_name (${config_rwo_sc_name})!"
      log_stderr "Either storage_type empty and both rwx_sc_name and rwo_sc_name full, or storage_type 'nfs-server' and rwo_sc_name full." 1
      ret 1
      handle_exit_status "Validating SC existing"
    fi
    
    if [[ -n "$config_rwx_sc_name" ]]; then
      run_command "kubectl get storageclasses ${config_rwx_sc_name} > /dev/null" "Validating if RWX StorageClass ${config_rwx_sc_name} exist"
    fi
    
    if [[ -n "$config_rwo_sc_name" ]]; then
      run_command "kubectl get storageclasses ${config_rwo_sc_name} > /dev/null " "Validating if RWO StorageClass ${config_rwo_sc_name} exist"
    fi
  fi
}

function validate_sufficient_pods() {
  if check_condition ${config_cluster_visibility} && ! check_condition ${config_skip_validations}; then
      local kubernetes_minimum_pods=58
      local node_count=$(yq '.k8s_nodes | length' ${VARS_FILE})
      local node_count_last=$(expr $node_count - 1)
      
      for i in $(seq 0 $node_count_last); do
        local node_name=$(yq ".k8s_nodes[${i}].node" ${VARS_FILE} | sed 's/"//g')
        local pod_capacity=$(kubectl get node ${node_name} -o jsonpath={.status.capacity.pods})

        log_to_sisense_installer "Validating node ${node_name} pods capacity"
        if [[ ${pod_capacity} -ge ${kubernetes_minimum_pods} ]]; then
          log_green "Node ${node_name} meets with the minimum requirements for pod capacity (minimum: ${kubernetes_minimum_pods}, current: ${pod_capacity})"
        else
          log_stderr "Node ${node_name} does not meet with the minimum requirements for pod capacity (minimum: ${kubernetes_minimum_pods}, current: ${pod_capacity})" 1
          handle_exit_status "Validate sufficient pods capacity"
        fi
      done
      handle_exit_status "Validate sufficient pods capacity"
  fi
}

function install_helmfile() {
  if ! command helmfile > /dev/null 2>&1; then
    if ! check_condition ${config_offline_installer}; then
      local HELMFILE_VERSION="0.159.0"
      local HELMFILE_URL="https://github.com/helmfile/helmfile/releases/download/v${HELMFILE_VERSION}/helmfile_${HELMFILE_VERSION}_linux_amd64.tar.gz"
      run_command "curl -sL ${HELMFILE_URL} | tar -xzO helmfile > ./helmfile" "Downloading Helmfile version v${HELMFILE_VERSION}"
      run_command "sudo mv ./helmfile /usr/local/bin/helmfile && sudo chmod +x /usr/local/bin/helmfile" "Moving Helmfile to /usr/local/bin/helmfile"

      if ! command -v git &> /dev/null ; then
        log_to_sisense_installer "Installing git before helmfile init (helmfile needs git)"
        pkg=$(which dnf 2>/dev/null || which yum 2>/dev/null || which apt 2>/dev/null)
        if [[ -z ${pkg} ]]; then
          log_stderr "Error: no package manager found in host." 1
          handle_exit_status "Git installation before helmfile init"
        fi
        run_command "sudo ${pkg} install git -y"
      fi

      run_command "helmfile init --force > /dev/null 2>&1" "Initializing Helmfile"
    else
      log_stderr "Error: Helmfile not installed, and is offline. Make sure to install Helmfile and run command 'helmfile init --force'." 1
      handle_exit_status "Installing Helmfile"
    fi
  fi
}

function patch_sisense_resources() {
  declare -A resources
  resources["deployment"]="api-gateway-maintenance"
  resources["service"]="api-gateway-maintenance"
  resources["ServiceAccount"]="api-gateway-maintenance"
  resources["ConfigMap"]="add-completion"

  local keys=("deployment" "service" "ServiceAccount" "ConfigMap")
  local is_resource_exist=""

  for resource in "${keys[@]}"; do
    resource_name="${resources[$resource]}"
    assign_to_helm_release ${resource} ${resource_name} ${config_namespace_name} ${config_namespace_name}
  done
}

function template_sisense() {
  local template_file=${TEMPLATES_DIR}/sisense-values.yaml.j2
  local values_file=${HELM_VALUES_DIR}/${config_namespace_name}-values.yaml
  local extra_values=${EXTRA_VALUES_HELM_DIR}/sisense.yml
  local all_values=/tmp/sisense_all_values.yaml

  log_to_sisense_installer "Getting Sisense extra values"
  # Unifying all variables with Sisense extra values
  yq -y -rs 'reduce .[] as $item ({}; . * $item)' ${UNIFIED_FILE} ${extra_values} > ${all_values}
  handle_exit_status "Getting Sisense extra values"

  if check_condition ${config_update} && [[ ${config_kubernetes_cloud_provider,,} == "aws" || ${config_cloud_provider,,} == "aws" ]] \
  && ! check_condition ${config_ebs_csi_enabled}; then
    aws_gp_sc=$(kubectl -n ${config_namespace_name} get pvc ${config_namespace_name}-grafana -ojsonpath='{.spec.storageClassName}' --ignore-not-found)
    if [[ -n ${aws_gp_sc} ]]; then
      log_to_sisense_installer "Keeping Sisense statefulSets PVCs with storage class ${aws_gp_sc}"
      echo "aws_gp_sc: ${aws_gp_sc}" >> ${all_values}
    fi
  fi

  if [[ ${config_storage_type} == "nfs" ]]; then
    nfs_sc=$(kubectl -n ${config_namespace_name} get pvc storage -ojsonpath='{.spec.storageClassName}' --ignore-not-found)
    nfs_sc=${nfs_sc:-$(kubectl get sc nfs-client -o jsonpath='{.metadata.name}')}
    if [[ -n ${nfs_sc} ]]; then
      log_to_sisense_installer "Keeping Sisense PVCs with storage class ${nfs_sc}"
      echo "nfs_sc: ${nfs_sc}" >> ${all_values}
    fi
  fi

  log_to_sisense_installer "Generating Sisense values file"
  template_j2_file ${template_file} ${all_values} ${values_file}

  helm_mapkubeapis ${config_namespace_name} ${config_namespace_name}
}

function template_prometheus() {
  if ! check_condition ${config_is_openshift} && check_condition ${config_cluster_visibility} \
    && check_condition ${config_internal_monitoring} && check_condition ${config_prometheus_enabled}; then

    local template_file=${TEMPLATES_DIR}/kube-prometheus-stack-values.yaml.j2
    local values_file=${LOGMON_DIR}/kube-prometheus-stack-values.yaml
    local extra_values=${EXTRA_VALUES_HELM_DIR}/kube-prometheus-stack.yml
    local all_values=/tmp/kube-prometheus-stack_all_values.yaml

    log_to_sisense_installer "Getting Prometheus extra values"
    # Unifying all variables with Prometheus extra values
    yq -y -rs 'reduce .[] as $item ({}; . * $item)' ${UNIFIED_FILE} ${extra_values} > ${all_values}
    handle_exit_status "Getting Prometheus extra values"

    if check_condition ${config_update} && [[ ${config_kubernetes_cloud_provider,,} == "aws" || ${config_cloud_provider,,} == "aws" ]] \
    && ! check_condition ${config_ebs_csi_enabled}; then
      aws_gp_sc=$(kubectl -n ${config_monitoring_namespace} get pvc -o jsonpath='{.items[0].spec.storageClassName}' --ignore-not-found)
      if [[ -n ${aws_gp_sc} ]]; then
        log_to_sisense_installer "Keeping Sisense Prometheus PVCs with storage class ${aws_gp_sc}"
        echo "aws_gp_sc: ${aws_gp_sc}" >> ${all_values}
      fi
    fi

    if [[ ${config_storage_type} == "nfs" ]]; then
      nfs_sc=$(kubectl -n ${config_monitoring_namespace} get pvc -o jsonpath='{.items[0].spec.storageClassName}' --ignore-not-found)
      nfs_sc=${nfs_sc:-$(kubectl get sc nfs-client -o jsonpath='{.metadata.name}')}
      if [[ -n ${nfs_sc} ]]; then
        log_to_sisense_installer "Keeping Sisense Prometheus PVCs with storage class ${nfs_sc}"
        echo "nfs_sc: ${nfs_sc}" >> ${all_values}
      fi
    fi

    export prometheus_disk_size=10
    export alertmanager_disk_size=1
    export grafana_disk_size=3

    log_to_sisense_installer "Generating Kube Prometheus Stack values file"
    template_j2_file ${template_file} ${all_values} ${values_file}

    if check_condition ${config_update} ; then
      log_to_sisense_installer "Upating CRD's for kube-prometheus-stack"
      run_command "kubectl apply --server-side -f ${FILES_DIR}/kube-prometheus-stack-crds --force-conflicts"
    fi

    if ! check_condition ${cluster_mode} || [[ ${config_storage_type,,} == "rook-ceph" ]]; then

      local alertmanager_pvc=$(kubectl get pv alertmanager-db --no-headers --ignore-not-found | wc -l)
      if [[ ${alertmanager_pvc} -eq 0 ]]; then
        template_file=${TEMPLATES_DIR}/alertmanager-pv.yaml.j2
        export alertmanager_pv_file=${LOGMON_DIR}/alertmanager-pv.yaml

        log_to_sisense_installer "Generating Alertmanager PV file"
        template_j2_file ${template_file} ${all_values} ${alertmanager_pv_file}
      fi

      local prometheus_pvc=$(kubectl get pv prometheus-db-prometheus-0 --no-headers --ignore-not-found | wc -l)
      if [[ ${prometheus_pvc} -eq 0 ]]; then
        template_file=${TEMPLATES_DIR}/prometheus-pv.yaml.j2
        export prometheus_pv_file=${LOGMON_DIR}/prometheus-pv.yaml

        log_to_sisense_installer "Generating Prometheus PV file"
        template_j2_file ${template_file} ${all_values} ${prometheus_pv_file}
      fi
    fi

    run_command "kubectl -n ${config_monitoring_namespace} delete ds cadvisor --ignore-not-found" "Deleting cAdvisor (ignoring not found)"
    helm_mapkubeapis sisense-prom-operator ${config_monitoring_namespace}
  fi
}

function template_logmon() {
  if ! check_condition ${config_is_openshift} && check_condition ${config_cluster_visibility} \
    && (check_condition ${config_internal_monitoring} || check_condition ${config_external_monitoring}); then

    local template_file=${TEMPLATES_DIR}/logging-monitoring-values.yaml.j2
    local values_file=${LOGMON_DIR}/logmon-values.yaml
    local extra_values=${EXTRA_VALUES_HELM_DIR}/logging.yml
    local all_values=/tmp/logmon_all_values.yaml

    log_to_sisense_installer "Getting Logging Monitoring extra values"
    # Unifying all variables with LogMon extra values
    yq -y -rs 'reduce .[] as $item ({}; . * $item)' ${UNIFIED_FILE} ${extra_values} > ${all_values}
    handle_exit_status "Getting Logging Monitoring extra values"

    log_to_sisense_installer "Generating Logging Monitoring values file"
    template_j2_file ${template_file} ${all_values} ${values_file}

    helm_mapkubeapis logmon-${config_namespace_name} ${config_namespace_name}
  fi
}

function template_cluster_metrics() {
  if check_condition ${config_external_monitoring}; then
    local template_file=${TEMPLATES_DIR}/cluster-metrics-values.yaml.j2
    local values_file=${LOGMON_DIR}/cluster-metrics-values.yaml
    local extra_values=${EXTRA_VALUES_HELM_DIR}/cluster-metrics.yml
    local all_values=/tmp/cluster-metrics_all_values.yaml

    log_to_sisense_installer "Getting Cluster Metrics extra values"
    # Unifying all variables with Cluster Metrics extra values
    yq -y -rs 'reduce .[] as $item ({}; . * $item)' ${UNIFIED_FILE} ${extra_values} > ${all_values}
    handle_exit_status "Getting Cluster Metrics extra values"

    log_to_sisense_installer "Generating Cluster Metrics values file"
    template_j2_file ${template_file} ${all_values} ${values_file}

    helm_mapkubeapis cluster-metrics ${config_monitoring_namespace}
  fi
}

function generate_tls_secret() {
  if check_condition ${config_is_ssl}; then
    log_to_sisense_installer "Generating TLS Secret"
    # Need to export for J2 template later (In template_nginx_ingress)
    export tls_secret_name=${config_tls_secret_name:-sisense-tls}
    local tls_secret_name_secondry=sisense-tls-secret

    if check_condition ${config_within_provisioner}; then
      if ! kubectl -n ${config_namespace_name} get secrets ${tls_secret_name} > /dev/null 2>&1 &&
         ! kubectl -n ${config_namespace_name} get secrets ${tls_secret_name_secondry} > /dev/null 2>&1; then
        log_stderr "No TLS secret named ${tls_secret_name} or ${tls_secret_name_secondry} found, please create it in advance" 1
        handle_exit_status "Validation Sisense TLS secret existence"
      else
        log_green "TLS secret found, continuing installation"
        return
      fi
    fi

    # maybe TODO - need to fix in provisioner, to create secret from files (add new values crt_file and key_file)
    log_to_sisense_installer "Creating secret ${tls_secret_name} out of ${config_ssl_key_path} and ${config_ssl_cer_path}"
    run_command "kubectl create namespace ${config_namespace_name} --dry-run=client -o yaml | kubectl apply -f -"
    run_command "kubectl delete secret ${tls_secret_name} -n ${config_namespace_name} --ignore-not-found"
    run_command "kubectl create secret tls ${tls_secret_name} -n ${config_namespace_name} --key=${config_ssl_key_path} --cert=${config_ssl_cer_path}"
  fi
}

function template_nginx_ingress(){
  if check_condition ${config_is_ssl} || ( ! check_condition ${config_is_ssl} && [[ ${config_gateway_port} -eq 80 ]] ); then
    local template_file=${TEMPLATES_DIR}/nginx-values.yaml.j2
    local values_file=${HELM_VALUES_DIR}/nginx-ingress-values.yaml
    local extra_values=${EXTRA_VALUES_HELM_DIR}/nginx-ingress.yml
    local all_values=/tmp/nginx-ingress-values.yaml

    log_to_sisense_installer "Getting Nginx-Ingress extra values"
    # Unifying all variables with Cluster Metrics extra values
    yq -y -rs 'reduce .[] as $item ({}; . * $item)' ${UNIFIED_FILE} ${extra_values} > ${all_values}
    handle_exit_status "Getting Nginx-Ingress  extra values"

    log_to_sisense_installer "Generating Nginx-Ingress values file"
    template_j2_file ${template_file} ${all_values} ${values_file}    

    helm_mapkubeapis nginx-ingress ${config_utils_namespace}
  fi
}

function template_cluster_autoscaler() {
  if check_condition ${config_cloud_auto_scaler} && [[ ${config_kubernetes_cloud_provider,,} != "azure" && ${config_kubernetes_cloud_provider,,} != "gke" ]]; then
    local template_file=${TEMPLATES_DIR}/cluster-autoscaler-values.yaml.j2
    local values_file=${HELM_VALUES_DIR}/cluster-autoscaler-values.yaml
    local extra_values=${EXTRA_VALUES_HELM_DIR}/cluster-autoscaler.yml
    local all_values=/tmp/cluster-autoscaler_all_values.yaml

    log_to_sisense_installer "Getting Cluster Autoscaler extra values"
    # Unifying all variables with Cluster Autoscaler extra values
    yq -y -rs 'reduce .[] as $item ({}; . * $item)' ${UNIFIED_FILE} ${extra_values} > ${all_values}
    handle_exit_status "Getting Cluster Autoscaler extra values"

    log_to_sisense_installer "Generating Cluster Autoscaler values file"
    template_j2_file ${template_file} ${all_values} ${values_file}

    helm_mapkubeapis cluster-autoscaler ${config_utils_namespace}
  fi
}

function template_alb_controller(){
  if check_condition ${config_alb_controller_enabled} && [[ ${config_kubernetes_cloud_provider} == "aws" ]]; then
    local template_file=${TEMPLATES_DIR}/alb-controller-values.yaml.j2
    local values_file=${HELM_VALUES_DIR}/alb-controller-values.yaml
    local extra_values=${EXTRA_VALUES_HELM_DIR}/alb-controller.yml
    local all_values=/tmp/alb-controller_all_values.yaml

    export alb_webhook_service=$(kubectl get svc aws-load-balancer-webhook-service -n ${config_utils_namespace} -o jsonpath={.spec.clusterIP} --ignore-not-found)
    log_to_sisense_installer "Getting ALB Controller extra values"
    # Unifying all variables with ALB Controller extra values
    yq -y -rs 'reduce .[] as $item ({}; . * $item)' ${UNIFIED_FILE} ${extra_values} > ${all_values}
    handle_exit_status "Getting Cluster Metrics extra values"

    log_to_sisense_installer "Generating ALB Controller values file"
    template_j2_file ${template_file} ${all_values} ${values_file}

    helm_mapkubeapis aws-load-balancer-controller ${config_utils_namespace}
  fi
}

function template_helmfile() {
  local template_file=${TEMPLATES_DIR}/helmfile.yaml.j2
  local helmfile_file=${HELM_VALUES_DIR}/helmfile.yaml
  export HELM_CHARTS_DIR=${WORK_DIR}/files

  log_to_sisense_installer "Generating Helmfile file"
  template_j2_file ${template_file} ${UNIFIED_FILE} ${helmfile_file}
}

function run_helmfile() {
  local helmfile_file=${HELM_VALUES_DIR}/helmfile.yaml
  log_to_sisense_installer "Deploying Sisense using helmfile with file ${helmfile_file}"
  run_command "helmfile --file ${helmfile_file} sync" "Deploying all Helm charts using Helmfile file ${helmfile_file}"
}

function generate_add_completion() {
  if [[ ! -f ~/add_completion-${config_namespace_name}.sh ]]; then
    log_to_sisense_installer "Generating the add-completion file ~/add_completion-${config_namespace_name}.sh"
    run_command "kubectl get cm --namespace ${config_namespace_name} add-completion -ojsonpath='{.data.*}' > ~/add_completion-${config_namespace_name}.sh"
    run_command "chmod u+x ~/add_completion-${config_namespace_name}.sh"
  fi
}

function enable_all_rabbitmq_feature_flags_before_upgrade(){
  # From rabbitmq 3.11.18 to 3.12.x all feature flags must be enabled before the upgrade
  if check_condition ${config_update}; then
    is_rabbitmq_running=$(kubectl -n ${config_namespace_name} get pod | grep -c ${config_namespace_name}-rabbitmq-0)
    if [[ ${is_rabbitmq_running} -gt 0 ]]; then
      log_to_sisense_installer "Enabling all RabbitMQ feature flags before upgrade"
      set -o pipefail
      kubectl exec -i ${config_namespace_name}-rabbitmq-0 -n ${config_namespace_name} -- rabbitmqctl enable_feature_flag all 2>&1 | tee -a sisense-installer.log 2>&1
      rc=$?
      set +o pipefail

      if [[ ${rc} -ne 0 ]]; then
        log_to_sisense_installer "Enabling RabbitMQ feature flags failed, restarting RabbitMQ and trying again"
        run_command "kubectl -n ${config_namespace_name} rollout restart sts ${config_namespace_name}-rabbitmq" "Running rollout restart for RabbitMQ"
        run_command "kubectl -n ${config_namespace_name} rollout status sts ${config_namespace_name}-rabbitmq --watch --timeout=900s" "Waiting for RabbitMQ to be ready..."
        run_command "kubectl exec -i ${config_namespace_name}-rabbitmq-0 -n ${config_namespace_name} -- rabbitmqctl enable_feature_flag all" "Enable all RabbitMQ feature flags before upgrade, second try"
      fi
    fi
  fi
}

function template_kyverno(){
  if check_condition ${config_signature_validation}; then
    local template_file=${TEMPLATES_DIR}/kyverno-values.yaml.j2
    local values_file=${HELM_VALUES_DIR}/kyverno-values.yaml
    local template_kyverno_cluster_policy=${TEMPLATES_DIR}/kyverno-cluster-policy.yaml.j2
    local kyverno_cluster_policy_file=${HELM_VALUES_DIR}/kyverno-cluster-policy.yaml

    log_to_sisense_installer "Generating Kyverno values file"
    template_j2_file ${template_file} ${UNIFIED_FILE} ${values_file}

    log_to_sisense_installer "Generating Kyverno clusterpolicy file"
    template_j2_file ${template_kyverno_cluster_policy} ${UNIFIED_FILE} ${kyverno_cluster_policy_file}
    run_command "kubectl create namespace kyverno --dry-run=client -o yaml | kubectl apply -f -"
    run_command "kubectl create secret generic cosign-public-key --from-file=cosign.pub=${FILES_DIR}/cosign.pub -n kyverno --dry-run=client -o yaml | kubectl apply -f -"
  fi
}
