#!/usr/bin/env bash

WORK_DIR=installer/06_storage_installation
TEMP_CONFIG=${WORK_DIR}/temp_config.yaml
REMOTE_VARS=${WORK_DIR}/remote_vars
REMOTE_SCRIPT=${WORK_DIR}/remote.sh
REMOTE_HOSTS=${WORK_DIR}/hosts
ERROR_DIR=${WORK_DIR}/errors
TEMPLATES_DIR=${WORK_DIR}/templates
FILES_DIR=${WORK_DIR}/files

source installer/common/functions.sh
source installer/06_storage_installation/patch_storage.sh

export pull_secrets_name=${config_pull_secrets_name:-sisense-pull-secret}

function patch_storage(){
  if ! check_condition "$cluster_mode"; then
    patch_single
  elif [[ ${config_storage_type} == "rook-ceph" ]]; then
    setup_manual_sc "Rook Ceph"
    patch_rook_ceph
  elif [[ ${config_storage_type} == "fsx" ]]; then
    patch_fsx
  else
    patch_pvc
  fi
}

function install_storage_drivers(){
  if ! check_condition ${cluster_mode}; then
    setup_manual_sc
  elif [[ ${config_storage_type} == "rook-ceph" ]]; then
    setup_manual_sc "Rook Ceph"
    install_rook_ceph
  elif [[ ${config_storage_type} == "fsx" ]]; then
    install_aws_ebs_csi_driver
    setup_gp3_sc
    install_fsx_driver
    setup_fsx_sc
    fsx_subpath
  elif [[ ${config_storage_type} == "azurefile" ]]; then
    if check_condition ${config_azure_csi_enabled}; then
      setup_azurefile_premium_sc
    else
      setup_azurefile_sc
    fi
  elif [[ ${config_storage_type} == "efs" ]]; then
    install_aws_ebs_csi_driver
    setup_gp3_sc
    install_efs_provisioner
  elif [[ ${config_storage_type} == "nfs" ]]; then
    install_aws_ebs_csi_driver
    setup_gp3_sc
    install_nfs_csi_driver
  fi
}

function install_efs_provisioner(){  
  EFS_PROVISIONER_TEMPLATE=${TEMPLATES_DIR}/efs-provisioner-values.yml.j2
  template_j2_file ${EFS_PROVISIONER_TEMPLATE} ${UNIFIED_FILE} ${FILES_DIR}/efs_provisioner_values.yml

  log_royal_blue "EFS Provisioner | Clearing deprecated objects in Helm release"
  helm_mapkubeapis efs-provisioner ${config_utils_namespace}

  efs_provisioner_version="0.14.0"
  efs_provisioner_extra_values=${EXTRA_VALUES_HELM_DIR}/efs.yml
  run_command "helm upgrade efs-provisioner ${FILES_DIR}/efs-provisioner-${efs_provisioner_version}.tgz --values ${FILES_DIR}/efs_provisioner_values.yml --values ${efs_provisioner_extra_values} --namespace ${config_utils_namespace} --install --create-namespace --cleanup-on-fail" "EFS | Install/Upgrade efs-provisioner Chart"

  log_royal_blue "Waiting for EFS Provisioner pods to be running"
  if ! are_all_pods_running "${config_utils_namespace}" "app" "efs-provisioner" 60 "Running"; then
    log_stderr "Error: EFS Provisioner pods failed to run" 1
    handle_exit_status "EFS Provisioner installation"
  fi

}
function install_aws_ebs_csi_driver(){
  local ebs_csi_installed=0
  local install_ebs_driver=0
  if [[ ${config_kubernetes_cloud_provider} == "aws" ]] || [[ ${config_cloud_provider} == "aws" ]] && ! check_condition ${config_offline_installer}; then
    ebs_csi_installed=$(kubectl get csidrivers.storage.k8s.io | grep -c ebs.csi.aws.com)

    if check_condition ${cluster_mode} && check_condition ${config_ebs_csi_enabled} ;then
      install_ebs_driver=1
    fi

    if ! check_condition "${config_ebs_csi_enabled}"; then
      if [ "${config_storage_type}" == "fsx" ] || [ "${config_storage_type}" == "efs" ] || [ "${config_storage_type}" == "nfs" ]; then
        install_ebs_driver=1
      fi
    fi
    
    if [[ ${ebs_csi_installed} -eq 0 && ${install_ebs_driver} -gt 0 ]]; then
      log_royal_blue "EBS CSI Driver | Clearing deprecated objects in Helm release"
      helm_mapkubeapis aws-ebs-csi-driver kube-system

      ebs_csi_driver_version="2.17.2"
      ebs_csi_extra_values=${EXTRA_VALUES_HELM_DIR}/aws-ebs-csi.yml
      run_command "helm upgrade aws-ebs-csi-driver ${FILES_DIR}/aws-ebs-csi-driver-${ebs_csi_driver_version}.tgz --values ${FILES_DIR}/ebs-csi-values.yaml --values ${ebs_csi_extra_values} --namespace kube-system --install --create-namespace --cleanup-on-fail" "EBS CSI | Install/Upgrade EBS CSI driver chart"

      log_royal_blue "Waiting for EBS CSI Driver pods to be running"
      if ! are_all_pods_running "kube-system" "app.kubernetes.io/name" "aws-ebs-csi-driver" 60 "Running"; then
        log_stderr "Error: EBS CSI Driver pods failed to run" 1
        handle_exit_status "EBS CSI Driver installation"
      fi

      setup_ebs_csi_sc

    fi
  fi

}

function setup_ebs_csi_sc(){
  export nodes_zones=$(kubectl get no -Lfailure-domain.beta.kubernetes.io/zone | awk 'NR>1 {print $6}' | sort | uniq)
  EBS_CSI_SC_TEMPLATE=${TEMPLATES_DIR}/ebs-csi-sc.yml.j2
  template_j2_file ${EBS_CSI_SC_TEMPLATE} ${UNIFIED_FILE} ${FILES_DIR}/ebs_csi_sc.yml
  run_command "kubectl apply -f ${FILES_DIR}/ebs_csi_sc.yml --force" "EBS CSI | Apply StorageClass"
}

function install_fsx_driver(){
  FSX_CSI_TEMPLATE=${TEMPLATES_DIR}/aws-fsx-csi-driver.yml.j2
  template_j2_file ${FSX_CSI_TEMPLATE} ${UNIFIED_FILE} ${FILES_DIR}/aws_fsx_csi_driver.yml
  run_command "kubectl apply -f ${FILES_DIR}/aws_fsx_csi_driver.yml --force" "FSX | Install FSX CSI Driver for Kubernetes"
}

function setup_fsx_sc(){
  run_command "kubectl apply -f ${FILES_DIR}/fsx-sc.yaml --force"
}

function fsx_subpath(){
  is_pvc_bound=$(kubectl get pvc -n ${config_namespace_name} storage 2>&1 | grep -c 'Bound')
  if [[ "${config_fsx_mount_name}" == *"/" ]] && [[ $is_pvc_bound == 0 ]]; then

    # create sisense ns to fsx-subpath job and pvc 
    run_command "kubectl create namespace ${config_namespace_name} --dry-run=client -o yaml | kubectl apply -f -"

    FSX_SUBPATH_TEMPLATE=${TEMPLATES_DIR}/fsx-subpath.yml.j2
    template_j2_file ${FSX_SUBPATH_TEMPLATE} ${UNIFIED_FILE} ${FILES_DIR}/fsx_subpath.yml
    run_command "kubectl apply -n ${config_namespace_name} -f ${FILES_DIR}/fsx_subpath.yml" "FSx Subpath | Deploy FSx subpath job"

    log_royal_blue "Waiting for FSx Subpath pods to complete"
    if ! are_all_pods_running "${config_namespace_name}" "app.kubernetes.io/name" "fsx-subpath" 20 "Succeeded"; then
      log_stderr "Error: FSx Subpath pods failed to complete" 1
      handle_exit_status "FSx Subpath installation"
    fi

    run_command "kubectl delete -n ${config_namespace_name} --force --grace-period=0 -f ${FILES_DIR}/fsx_subpath.yml 2>&1" "FSx Subpath | Terminate FSx subpath job"
  fi
}

function setup_gp3_sc(){
  if ([[ ${config_kubernetes_cloud_provider} == "aws" ]] || [[ ${config_cloud_provider} == "aws" ]]) && ! check_condition ${config_ebs_csi_enabled} && ! check_condition ${config_offline_installer}; then
    export nodes_zones=$(kubectl get no -Lfailure-domain.beta.kubernetes.io/zone | awk 'NR>1 {print $6}' | sort | uniq)
    GP2_SC_TEMPLATE=${TEMPLATES_DIR}/gp3-sc.yml.j2
    template_j2_file ${GP2_SC_TEMPLATE} ${UNIFIED_FILE} ${FILES_DIR}/gp3_sc.yml
    run_command "kubectl apply -f ${FILES_DIR}/gp3_sc.yml --force" "GP2 | Apply StorageClass"
  fi
}

function setup_azurefile_sc(){
  AZUREFILE_SC_TEMPLATE=${TEMPLATES_DIR}/azurefile-sc-rbac.yml.j2
  template_j2_file ${AZUREFILE_SC_TEMPLATE} ${UNIFIED_FILE} ${FILES_DIR}/azurefile_sc_rbac.yml
  run_command "kubectl apply -f ${FILES_DIR}/azurefile_sc_rbac.yml --force" "Azurefile | Deploy Azurefile StorageClass and RBAC"
}

function setup_azurefile_premium_sc(){
  local is_azurefile_csi=$(kubectl get -n kube-system daemonset | grep -c "csi-azuredisk-node")

  if [[ ${is_azurefile_csi} -eq 0 ]]; then
    echo "Azure CSI DaemonSet is missing, Please enable CSI for your AKS https://docs.microsoft.com/en-us/azure/aks/csi-storage-drivers"
    exit 1
  fi
  AZUREFILE_SC_TEMPLATE=${TEMPLATES_DIR}/azurefile-csi-premium-sc.yml.j2
  template_j2_file ${AZUREFILE_SC_TEMPLATE} ${UNIFIED_FILE} ${FILES_DIR}/azurefile_csi_premium_sc.yml
  run_command "kubectl apply -f ${FILES_DIR}/azurefile_csi_premium_sc.yml --force" "Azurefile | Deploy Azurefile csi premium StorageClass"
}

function setup_manual_sc(){
  deployment_type=${1:-Single}
  if ! kubectl get sc manual > /dev/null 2>&1; then
    run_command "kubectl apply -f ${FILES_DIR}/manual-sc.yaml" "${deployment_type} | Deploy Manual StorageClass"
  fi
}

#### Needs to decide if planning to use it
function install_nfs_csi_driver(){
  local nfs_csi_namespace=${config_utils_namespace}
  local nfs_csi_release=csi-driver-nfs
  local nfs_csi_driver_version="4.12.1"
  local nfs_csi_template=${TEMPLATES_DIR}/nfs-csi-values.yml.j2
  local nfs_csi_values_file=${FILES_DIR}/nfs-csi-values.yml
  local nfs_client_installed=$(kubectl get sc | grep -c nfs-client)

  if [[ ${nfs_client_installed} -eq 0 ]]; then
    log_royal_blue "NFS CSI Driver | Clearing deprecated objects in Helm release"
    helm_mapkubeapis ${nfs_csi_release} ${nfs_csi_namespace}

    template_j2_file ${nfs_csi_template} ${UNIFIED_FILE} ${nfs_csi_values_file}
    run_command "helm upgrade -n ${nfs_csi_namespace} ${nfs_csi_release} ${FILES_DIR}/csi-driver-nfs-${nfs_csi_driver_version}.tgz --values ${nfs_csi_values_file} --install --create-namespace --cleanup-on-fail" "Install/Upgrade NFS CSI driver chart"

    log_royal_blue "Waiting for NFS CSI Driver pods to be running"
    if ! are_all_pods_running "${nfs_csi_namespace}" "app.kubernetes.io/instance" "${nfs_csi_release}" 60 "Running"; then
      log_stderr "Error: NFS CSI Driver pods failed to run" 1
      handle_exit_status "NFS CSI Driver installation"
    fi
  fi
}

function install_rook_ceph() {
  local files_dir=${FILES_DIR}/rook-ceph
  local rook_ceph_dir=${sisense_dir}/config/rook-ceph
  local rook_ceph_version=v1.11.9
  local rook_ceph_release=rook-operator
  # Exported since it's part of the j2 template file later on
  export rook_ceph_namespace=rook-ceph

  local attempt=0
  local retries=100

  run_command "mkdir -p ${rook_ceph_dir}"

  #######################################################
  ### 1st part: Helm chart Rook Operator installation ###
  #######################################################
  local is_rook_ceph_installed=$(helm ls -n ${rook_ceph_namespace} | grep ${rook_ceph_release})
  if [[ -z ${is_rook_ceph_installed} ]]; then
    local kubernetes_version=$(kubectl version --short -ojson 2>/dev/null | jq '.serverVersion.gitVersion' | sed 's/"//g')
    local node_names=$(yq '.k8s_nodes[].node' ${UNIFIED_FILE} | sed 's/"//g' | tr '\n' ' ')
    local psp_enabled=false
    if version_lt ${kubernetes_version} v1.25; then psp_enabled=true; fi

    log_royal_blue "Rook Ceph | Clearing deprecated objects in Helm release"
    helm_mapkubeapis ${rook_ceph_release} ${rook_ceph_namespace}

    run_command "kubectl label nodes ${node_names} role=storage-node --overwrite=true" "Labeling nodes ${node_names} as Rook Ceph storage"

    log_to_sisense_installer "Installing Rook Ceph Helm chart version ${rook_ceph_version} on namespace ${rook_ceph_namespace}"
    helm upgrade ${rook_ceph_release}  \
      ${files_dir}/rook-ceph-${rook_ceph_version}.tgz \
      --namespace ${rook_ceph_namespace} \
      --install \
      --create-namespace \
      --cleanup-on-fail \
      --set csi.pluginNodeAffinity='role=storage-node' \
      --set csi.provisionerNodeAffinity='role=storage-node' \
      --set pspEnable=${psp_enabled}
    handle_exit_status "Installing Rook Ceph Helm chart version ${rook_ceph_version} on namespace ${rook_ceph_namespace}"

    log_royal_blue "Waiting for Rook Ceph Operator pods to be running"
    if ! are_all_pods_running "${rook_ceph_namespace}" "app" "rook-ceph-operator" 60 "Running"; then
      log_stderr "Error: Rook Ceph Operator pods are not running" 1
      handle_exit_status "Rook Ceph installation"
    fi
  else
    log_green "Rook Ceph Helm chart already installed."
  fi
  # End 1st part

  ##########################################
  ### 2nd part: CephCluster installation ###
  ##########################################
  # Need to export for later j2 template
  export ceph_cluster_name=rook-ceph
  local is_ceph_cluster_exist=$(kubectl -n ${rook_ceph_namespace} get CephCluster ${ceph_cluster_name} --no-headers --ignore-not-found=true)
  if [[ -z ${is_ceph_cluster_exist} ]]; then
    local cluster_template=${TEMPLATES_DIR}/ceph-cluster.yaml.j2
    local cluster_file=${rook_ceph_dir}/ceph-cluster.yaml
  
    template_j2_file ${cluster_template} ${UNIFIED_FILE} ${cluster_file}
    run_command "kubectl apply -f ${cluster_file}" "Creating CephCluster ${ceph_cluster_name} in namespace ${rook_ceph_namespace}"

    log_royal_blue "Waiting for CephCluster to be created"
    local ceph_cluster_status=$(kubectl -n ${rook_ceph_namespace} get CephCluster ${ceph_cluster_name} -o jsonpath='{.status.state}')
    attempt=0
    retries=100

    while [[ ${ceph_cluster_status} != "Created" && ${attempt} -lt ${retries} ]]; do
      sleep 5
      ceph_cluster_status=$(kubectl -n ${rook_ceph_namespace} get CephCluster ${ceph_cluster_name} -o jsonpath='{.status.state}')
      attempt=$(expr ${attempt} + 1)
      log_royal_blue "Waiting for CephCluster to be created ... Attempt ${attempt}/${retries}"
    done
    if [[ ${ceph_cluster_status} == "Created" ]]; then
      log_green "CephCluster created successfully."
    else
      log_stderr "Error: CephCluster failed to create in time." 1
      handle_exit_status "CephCluster creation"
    fi
  else
    log_green "CephCluster already existing."
  fi
  # End 2nd part

  ############################################
  ### 3rd part: CephBlockPool installation ###
  ############################################
  local blockpool_name=replicapool
  local is_blockpool_exist=$(kubectl -n ${rook_ceph_namespace} get CephBlockPool ${blockpool_name} --no-headers  --ignore-not-found=true)
  if [[ -z ${is_blockpool_exist} ]]; then
    run_command "kubectl apply -f ${files_dir}/block-pool.yaml" "Creating CephBlockPool ${blockpool_name} in namespace ${rook_ceph_namespace}"

    log_royal_blue "Waiting for CephBlockPool to be ready"
    local blockpool_status=$(kubectl -n ${rook_ceph_namespace} get CephBlockPool ${blockpool_name} -o jsonpath='{.status.phase}')
    attempt=0
    retries=100

    while [[ ${blockpool_status} != "Ready" && ${attempt} -lt ${retries} ]]; do
      sleep 5
      blockpool_status=$(kubectl -n ${rook_ceph_namespace} get CephBlockPool ${blockpool_name} -o jsonpath='{.status.phase}')
      attempt=$(expr ${attempt} + 1)
      log_royal_blue "Waiting for CephBlockPool to be ready ... Attempt ${attempt}/${retries}"
    done
    if [[ ${blockpool_status} == "Ready" ]]; then
      log_green "CephBlockPool created successfully and ready."
    else
      log_stderr "Error: CephBlockPool failed to create in time." 1
      handle_exit_status "CephBlockPool creation"
    fi
  else
    log_green "CephBlockPool already existing."
  fi
  # End 3rd part

  #############################################
  ### 4th part: CephFilesystem installation ###
  #############################################
  local file_system_name=rwx-fs
  local is_file_system_exist=$(kubectl -n ${rook_ceph_namespace} get CephFilesystem ${file_system_name} --no-headers  --ignore-not-found=true)
  if [[ -z ${is_file_system_exist} ]]; then
    run_command "kubectl apply -f ${files_dir}/filesystem.yaml" "Creating CephFilesystem ${file_system_name} in namespace ${rook_ceph_namespace}"

    log_royal_blue "Waiting for CephFilesystem to be ready"
    local file_system_status=$(kubectl -n ${rook_ceph_namespace} get CephFilesystem ${file_system_name} -o jsonpath='{.status.phase}')
    attempt=0
    retries=100

    while [[ ${file_system_status} != "Ready" && ${attempt} -lt ${retries} ]]; do
      sleep 5
      file_system_status=$(kubectl -n ${rook_ceph_namespace} get CephFilesystem ${file_system_name} -o jsonpath='{.status.phase}')
      attempt=$(expr ${attempt} + 1)
      log_royal_blue "Waiting for CephFilesystem to be ready ... Attempt ${attempt}/${retries}"
    done
    if [[ ${file_system_status} == "Ready" ]]; then
      log_green "CephFilesystem created successfully and ready."
    else
      log_stderr "Error: CephFilesystem failed to create in time." 1
      handle_exit_status "CephFilesystem creation"
    fi
  else
    log_green "CephFilesystem already existing."
  fi
  # End 4th part

  #######################################
  ### 5th part: StorageClass creation ###
  #######################################
  local sc_cephfs=rook-cephfs-r3
  local sc_block=rook-ceph-block
  local is_sc_cephfs_exist=$(kubectl get sc ${sc_cephfs} --ignore-not-found=true)
  local is_sc_block_exist=$(kubectl get sc ${sc_block} --ignore-not-found=true)

  if [[ -z ${is_sc_cephfs_exist} || -z ${is_sc_block_exist} ]]; then
    run_command "kubectl apply -f ${files_dir}/storageclass.yaml --force" "Creating StorageClasses ${sc_cephfs} and ${sc_block}"

    is_sc_cephfs_exist=$(kubectl get sc ${sc_cephfs} --ignore-not-found=true)
    is_sc_block_exist=$(kubectl get sc ${sc_block} --ignore-not-found=true)
    if [[ -z ${is_sc_cephfs_exist} || -z ${is_sc_block_exist} ]]; then
      log_stderr "Error: StorageClass ${sc_cephfs} or ${sc_block} was not created." 1
      handle_exit_status "Rook Ceph StorageClasses creation"
    else
      log_green "StorageClasses ${sc_cephfs} and ${sc_block} created successfully."
    fi
  else
    log_green "Rook Ceph StorageClasses already existing."
  fi
}
