#!/bin/bash

WORK_DIR=installer/05_post_infra
TEMP_CONFIG=${WORK_DIR}/temp_config.yaml
REMOTE_VARS=${WORK_DIR}/remote_vars
REMOTE_SCRIPT=${WORK_DIR}/remote.sh
REMOTE_HOSTS=${WORK_DIR}/hosts
ERROR_DIR=${WORK_DIR}/errors
TEMPLATES_DIR=${WORK_DIR}/templates
FILES_DIR=${WORK_DIR}/files

KUBECTL_VERSION=v1.34.1
HELM_VERSION=v3.16.4
KUBECTL_URL="https://dl.k8s.io/release/${KUBECTL_VERSION}/bin/linux/amd64/kubectl"
HELM_URL="https://get.helm.sh/helm-${HELM_VERSION}-linux-amd64.tar.gz"

source installer/common/functions.sh

function get_binaries() {
  log_to_sisense_installer "Getting binaries kubectl (${KUBECTL_VERSION}) and helm (${HELM_VERSION})"
  run_command "mkdir -p ${FILES_DIR}"

  if check_condition ${config_offline_installer}; then
    log_to_sisense_installer "Taking them from the offline container"
    local kubectl_bin=""
    local helm_bin=""
    kubectl_bin=$(which kubectl)
    helm_bin=$(which helm)
    run_command "cp ${kubectl_bin} ${helm_bin} ${FILES_DIR}/"
  else
    log_to_sisense_installer "Downloading them from the internet"
    run_command "curl -L ${KUBECTL_URL} -o ${FILES_DIR}/kubectl"
    run_command "curl -L ${HELM_URL} -o ${FILES_DIR}/helm.tar.gz"
    run_command "tar -xvf ${FILES_DIR}/helm.tar.gz -C ${FILES_DIR} linux-amd64/helm && mv ${FILES_DIR}/linux-amd64/helm ${FILES_DIR}/helm"
  fi
  run_command "chmod -R 0755 ${FILES_DIR}"
  run_command "sudo cp ${FILES_DIR}/kubectl ${FILES_DIR}/helm /usr/local/bin"
}

function get_helm_mapkubeapis() {
  local mapkubeapis_version="0.4.1"
  if ! helm plugin ls | grep -q mapkubeapis; then
    log_to_sisense_installer "Installing Helm plugin mapkubeapis version ${mapkubeapis_version}"

    eval $(helm env) || handle_exit_status "command helm env"
    mkdir -p $HELM_DATA_HOME/plugins || handle_exit_status "mkdir -p $HELM_DATA_HOME/plugins"
    tar -xvf ${FILES_DIR}/helm-mapkubeapis-${mapkubeapis_version}.tgz --directory $HELM_DATA_HOME/plugins/ > /dev/null
    handle_exit_status "command tar -xvf ${FILES_DIR}/helm-mapkubeapis-${mapkubeapis_version}.tgz"

    helm plugin ls | grep -q mapkubeapis
    if [[ $? -eq 0 ]]; then
      log_green "Helm plugin mapkubeapis version ${mapkubeapis_version} successfully installed."
    else
      log_stderr "Something went wrong while installing Helm plugin mapkubeapis." 2
    fi
  else
    log_green "Helm plugin mapkubeapis already installed."
  fi
}

function check_bash_completion() {
  # Check if the bash-completion package is installed
  local package_name="bash-completion"

  if command -v dpkg >/dev/null && dpkg -l "$package_name" | grep -q "^ii"; then
    # For ubuntu
    return 0
  elif command -v rpm >/dev/null && rpm -q "$package_name" >/dev/null 2>&1; then
    # for RedHat/Rocky/Amazon
    return 0
  else
    log_royal_blue "Bash-completion package not found."
    log_royal_blue "Please install it, if you wish to enjoy auto complete kubectl and helm commands."
    return 1
  fi
}

function kubectl_auto_completion() {
  if ! check_condition ${config_offline_installer}; then
    pkg=$(which dnf 2>/dev/null || which yum 2>/dev/null || which apt 2>/dev/null)
    if [[ -z ${pkg} ]]; then
      log_stderr "Error: no package manager found in host." 1
      handle_exit_status "installing bash-completion"
    fi
    run_command "sudo ${pkg} install bash-completion -y" "Installing bash-completion"
  fi

  # Just incase the offline env already has bash-completion, 
  # so the user could enjoy the auto complete kubectl+helm commands :)
  if check_bash_completion; then
    log_to_sisense_installer "Adding kubectl and helm auto completion"
    if ! grep 'source <(kubectl completion bash) 2>/dev/null' ~/.bashrc; then
      echo 'source <(kubectl completion bash) 2>/dev/null' >> ~/.bashrc
      handle_exit_status "Adding kubectl auto completion"
    fi

    if ! grep 'source <(helm completion bash) 2>/dev/null' ~/.bashrc; then
      echo 'source <(helm completion bash) 2>/dev/null' >> ~/.bashrc
      handle_exit_status "Adding helm auto completion"
    fi

    if check_condition ${config_is_openshift} && ! grep 'source <(oc completion bash) 2>/dev/null' ~/.bashrc; then
      echo 'source <(oc completion bash) 2>/dev/null' >> ~/.bashrc
      handle_exit_status "Adding oc auto completion"
    fi
  fi
}

function post_rke() {
  generate_remote_vars "post RKE"
  local copy_msg="Copying kubectl (${KUBECTL_VERSION}) and helm (${HELM_VERSION})"
  local move_msg="Moving kubectl and helm into /usr/local/bin"
  local stuff_to_copy="${FILES_DIR}/kubectl ${FILES_DIR}/helm"

  log_to_sisense_installer "${copy_msg} to all nodes"

  for host in $(cat ${REMOTE_HOSTS}); do
    if [[ -n ${config_password} ]]; then
      log_to_sisense_installer "${copy_msg} to ${config_linux_user}@${host} using password"
      sshpass -p "${config_password}" scp -o "StrictHostKeyChecking no" -r ${stuff_to_copy} ${config_linux_user}@${host}:~
      handle_exit_status "${copy_msg} to ${config_linux_user}@${host} using password"

      log_to_sisense_installer "${move_msg} at ${config_linux_user}@${host} using password"
      sshpass -p "${config_password}" ssh -o "StrictHostKeyChecking no" "${config_linux_user}@${host}" sudo mv ~/kubectl ~/helm /usr/local/bin
      handle_exit_status "${move_msg} at ${config_linux_user}@${host} using password"
    else
      log_to_sisense_installer "${copy_msg} to to ${config_linux_user}@${host} using SSH key"
      scp -i ${config_ssh_key} -o "StrictHostKeyChecking no" -r ${stuff_to_copy} ${config_linux_user}@${host}:~
      handle_exit_status "${copy_msg} to ${config_linux_user}@${host} using SSH key"

      log_to_sisense_installer "${move_msg} at ${config_linux_user}@${host} using SSH key"
      ssh -i ${config_ssh_key} -o "StrictHostKeyChecking no" "${config_linux_user}@${host}" sudo mv ~/kubectl ~/helm /usr/local/bin
      handle_exit_status "${move_msg} at ${config_linux_user}@${host} SSH key"
    fi
  done

  log_to_sisense_installer "Validating K8s nodes are up"
  are_all_nodes_ready
  handle_exit_status "validating K8s nodes are up"

  log_green "Kubernetes cluster is up and running."
}

function aws_cloud_controller_manager() {
  # based on https://rke.docs.rancher.com/config-options/cloud-providers/aws#using-the-out-of-tree-aws-cloud-provider-for-rke
  if [[ ${config_cloud_provider,,} == "aws" && need_infra_installation ]]; then
    local values_file=${FILES_DIR}/aws_cloud_controller_values.yaml
    log_to_sisense_installer "Creating daemonset aws-cloud-controller-manager"
    run_command "helm repo add aws-cloud-controller-manager https://kubernetes.github.io/cloud-provider-aws"
    run_command "helm repo update"
    helm_mapkubeapis "aws-cloud-controller-manager" "kube-system"
    run_command "helm upgrade --install aws-cloud-controller-manager -n kube-system aws-cloud-controller-manager/aws-cloud-controller-manager --values ${values_file}"

    log_to_sisense_installer "Patch daemonset aws-cloud-controller-manager"
    run_command "kubectl -n kube-system patch ds aws-cloud-controller-manager --type=json -p=\"[{'op': 'remove', 'path': '/spec/template/spec/nodeSelector/node-role.kubernetes.io~1control-plane'}]\""
    run_command "kubectl rollout status ds -n kube-system aws-cloud-controller-manager"
  fi
}

function generate_docker_pull_secret() {
  export pull_secrets_name=${config_pull_secrets_name:-sisense-pull-secret}

  if [[ -n ${config_docker_registry} && -n ${config_docker_username} && -n ${config_docker_password} ]]; then
    list_of_namespaces="${config_namespace_name} ${config_utils_namespace}"
    if check_condition ${config_internal_monitoring} ||check_condition ${config_external_monitoring}; then
      list_of_namespaces="${list_of_namespaces} ${config_monitoring_namespace}"
    fi

    ## for testing kyverno
    if check_condition ${config_signature_validation}; then
      list_of_namespaces="${list_of_namespaces} kyverno"
    fi
    
    for ns in ${list_of_namespaces}; do
      run_command "kubectl create namespace ${ns} --dry-run=client -o yaml | kubectl apply -f -"

      is_secret_exist=$(kubectl -n ${ns} get secrets ${pull_secrets_name} --ignore-not-found)
      if check_condition ${config_overwrite_docker_secret} || [[ -z ${is_secret_exist} ]]; then
        log_to_sisense_installer "Generating Docker pull secrets for private docker registry ${config_docker_registry} in namespace '${ns}'"
        run_command "kubectl -n ${ns} create secret docker-registry ${pull_secrets_name} --docker-server=${config_docker_registry} --docker-username=${config_docker_username} --docker-password=${config_docker_password} --dry-run=client -o yaml --save-config | kubectl apply -f -"
      fi
    done
  fi

  export is_secret_exist=$(kubectl -n ${config_namespace_name} get secrets ${pull_secrets_name} --ignore-not-found)
}
