#!/usr/bin/env bash

# This script runs on all remote k8s nodes
# It runs all sorts of prerequisites beofre RKE installation
# Such as OS bootstrap, Docker installation, etc...

#!/bin/bash
DEFAULT_KERNEL_MODULES="br_netfilter,ip6_udp_tunnel,ip_set,ip_set_hash_ip,ip_set_hash_net,iptable_filter,iptable_nat,iptable_mangle,iptable_raw,nf_conntrack_netlink,nf_conntrack,nf_defrag_ipv4,nf_nat,nf_tables,udp_tunnel,veth,vxlan,x_tables,xt_addrtype,xt_conntrack,xt_comment,xt_mark,xt_multiport,xt_nat,xt_recent,xt_set,xt_statistic,xt_tcpudp"
IPVS_KERNEL_MODULES="ip_vs,ip_vs_rr,ip_vs_wrr,ip_vs_sh"

function release_dpkg_lock() {
  #TODO - maybe change function name
  if ! check_condition "$config_offline_installer"; then
  if [[ ${OS} == "Ubuntu" ]] || [[ ${OS} == "Debian" ]]; then
    if [ -f "/var/lib/dpkg/lock" ]; then
    log "detected lock on dpkg, releasing it"
    sudo rm -f /var/lib/dpkg/lock
    handle_exit_status "release_dpkg_lock"
    fi
  elif [[ ${OS} == *"Red Hat"* ]] || [[ ${OS} == *"CentOS"* ]] || [[ ${OS} == *"Rocky"* ]]; then
    if [ -f /var/run/yum.pid ]; then
    log "detected lock on yum, releasing it"
    sudo rm -rf /var/run/yum.pid
    handle_exit_status "release_dpkg_lock"
    fi
  fi
  fi
}

function bootstrap_centos() {
  local centos_fastestmirror_enabled=false
  
  # Add proxy to yum.conf or dnf.conf if http_proxy is defined
  if [[ -n "$http_proxy" ]]; then
    config_path="/etc/dnf/dnf.conf"
    ini_file_set "$config_path" "main" "proxy" "$http_proxy" 0644 sudo log
  fi

  if [[ -n "$http_proxy" ]] && [[ ${OS} == *"Red Hat"* ]]; then
    local proxy_hostname
    local proxy_port
    proxy_hostname=$(echo "$http_proxy" | sed -e 's|^http://||' -e 's|^https://||' | awk -F ':' '{print $1}')
    proxy_port=$(echo "$http_proxy" | sed -e 's/^.*://')
    run_command "sudo /sbin/subscription-manager config --server.proxy_hostname=${proxy_hostname} --server.proxy_port=${proxy_port}" "Adding proxy to RHEL subscription-manager"
  fi

  # Check presence of fastestmirror.conf
  if [[ -f "/etc/yum/pluginconf.d/fastestmirror.conf" ]] && [[ "${centos_fastestmirror_enabled,,}" == "false" ]]; then
    log "Disabling fastestmirror.conf"
    # Disable fastestmirror plugin if requested
    sudo sed -i 's/^enabled=.*/enabled=0/' /etc/yum/pluginconf.d/fastestmirror.conf
    handle_exit_status "disable fastestmirror.conf"
  fi

  # Install libselinux python package
  package_name="python3-libselinux"
  run_command "sudo yum -y install $package_name"
  handle_exit_status "bootstrap_centos"
}

function bootstrap_ubuntu() {
  
  # Check if bootstrap is needed
  which_python3=$(which python3)
  need_bootstrap_rc=$?

  # Check http::proxy in apt configuration files
  apt-config dump | grep -qsi 'Acquire::http::proxy'
  need_http_proxy_rc=$?

  # Check https::proxy in apt configuration files
  apt-config dump | grep -qsi 'Acquire::https::proxy'
  need_https_proxy_rc=$?

  # Check OS version
  os_version=$(cat /etc/os-release)

  if [[ -n ${http_proxy} ]] && [[ ${need_http_proxy_rc} -ne 0 ]]; then
    log "Add http_proxy ${http_proxy} to /etc/apt/apt.conf since http_proxy is defined"
    echo "Acquire::http::proxy \"$http_proxy\";" | sudo tee -a /etc/apt/apt.conf > /dev/null
    handle_exit_status "Add http_proxy to /etc/apt/apt.conf"
  fi

  if [[ -n ${https_proxy} ]] && [[ ${need_https_proxy_rc} -ne 0 ]]; then
    log "Add https_proxy ${https_proxy} to /etc/apt/apt.conf since https_proxy is defined"
    echo "Acquire::https::proxy \"$https_proxy\";" | sudo tee -a /etc/apt/apt.conf > /dev/null
    handle_exit_status "Add https_proxy to /etc/apt/apt.conf"
  fi

  if [[ ${need_bootstrap_rc} -ne 0 ]]; then
    run_command "sudo apt-get update" "apt-get update"
    run_command "sudo DEBIAN_FRONTEND=noninteractive apt-get install -y python3-minimal" "Install python3 (Ubuntu)"
  fi

  if [[ "$os_version" == *"ID=debian"* && ( "$VER" == "10"* || "$VER" == "11"* ) ]]; then
    run_command "sudo apt-get update --allow-releaseinfo-change" "Update Apt cache"
  fi

  # Install dbus for the hostname module
  # Workaround for https://github.com/ansible/ansible/issues/25543
  run_command "sudo DEBIAN_FRONTEND=noninteractive apt-get install -y dbus" "Install dbus for the hostname module"
}

function bootstrap_os() {
  if [[ ${OS} == *"CentOS"* ]] || [[ ${OS} == *"Rocky"* ]] || [[ ${OS} == *"Red Hat"* ]]; then
    bootstrap_centos
  elif [[ ${OS} == "Ubuntu" ]]; then
    bootstrap_ubuntu  
  fi

  log "Ensure bash_completion.d folder exists"
  if [ ! -d "/etc/bash_completion.d/" ]; then
    run_command "sudo mkdir -p /etc/bash_completion.d/"
    run_command "sudo chown root:root /etc/bash_completion.d/"
    run_command "sudo chmod 0755 /etc/bash_completion.d/"
  fi
}

function disable_swap() {
  log "Disabling swap on all nodes ..."
  run_command "sudo swapoff -a"
  run_command "sudo sed -i '/swap/d' /etc/fstab"
  handle_exit_status "disable swap"
}

function disable_seliux() {
  log "Disabling SELinux..."
  # For example: Ubuntu 22 by default don't have "setenforce"
  run_command "sudo setenforce 0 || true"
  if [[ -f /etc/selinux/config ]]; then
    if grep ^SELINUX= /etc/selinux/config >> /dev/null; then
      run_command "sudo sed -i 's/^SELINUX=.*/SELINUX=disabled/' /etc/selinux/config"
    else
      run_command "echo SELINUX=disabled | sudo tee -a /etc/selinux/config > /dev/null"
    fi
  fi
  handle_exit_status "disable SELinux"
}

function enable_kernel_modules() {
  local module_list="$1"
  IFS=',' read -ra modules <<< "$module_list"

  for module in "${modules[@]}"; do
    log "Enabling module: $module"
    run_command "sudo modprobe $module"
    handle_exit_status "enable module: $module"
  done
}

function modify_sysctl_entries() {
  log "Modifying sysctl entries ..."

  declare -A entries=(
    ["net.bridge.bridge-nf-call-ip6tables"]="1"
    ["net.bridge.bridge-nf-call-iptables"]="1"
    ["net.ipv4.ip_forward"]="1"
  )
  set_sysctl_entries /etc/sysctl.conf
  handle_exit_status "modifying sysctl entries"
}

function set_user_namespaces_sysctl() {
  log "Setting user namespaces sysctl value"

  declare -A entries=(
    ["user.max_user_namespaces"]="15064"
  )
  set_sysctl_entries /etc/sysctl.d/00-namespaces.conf
  handle_exit_status "setting user namespaces sysctl value"
}


function configure_kernel_runtime_parameters() {
  log "Configuring kernel runtime parameters"

  declare -A entries=(
    ["vm.overcommit_memory"]="1"
    ["vm.panic_on_oom"]="0"
    ["kernel.panic"]="10"
    ["kernel.panic_on_oops"]="1"
    ["kernel.keys.root_maxbytes"]="25000000"
  )
  set_sysctl_entries /etc/sysctl.d/90-kubelet.conf
  handle_exit_status "configuring kernel runtime parameters"
}

function create_kubernetes_manifests_folder() {
  local manifests_folder="/etc/kubernetes/manifests"
  log "Ensuring Kubernetes manifests folder ${manifests_folder} exists"

  run_command "sudo mkdir -p ${manifests_folder}"
  run_command "sudo chmod 0755 ${manifests_folder}"
  run_command "sudo chown root:root ${manifests_folder}"
}

function ensure_sisense_directories() {
  run_as_user=${run_as_user:-1000}
  run_as_group=${run_as_group:-1000}

  local dirs=("zookeeper"
    "mongodb"
    "storage"
    "dgraph-io"
    "prometheus"
    "alertmanager"
    "grafana"
    "grafana/grafana-${namespace_name}"
    "prometheus/prometheus-db"
    "alertmanager/alertmanager-db"
    "config"
  )

  log "Ensuring application directories exist with correct ownership (${run_as_user}:${run_as_group})"
  for dir in "${dirs[@]}"; do
    local current_path="${sisense_dir}/${dir}"
    log "Ensuring directory ${current_path}"
    run_command "sudo mkdir -p ${current_path}"
  done

  if ! check_condition ${skip_chown_sisense}; then
    local verbose=""
    if check_condition ${chown_sisense_verbose}; then
      log "Setting verbose to chown command"
      verbose="-v"
    fi
    if check_condition ${chown_sisense_dirs_only}; then
      log "Fixing ${sisense_dir} ownership to directories only"
      run_command "sudo find ${sisense_dir} -type d -exec chown ${run_as_user}:${run_as_group} ${verbose} {} +"
    else
      log "Fixing ${sisense_dir} ownership to all files"
      run_command "sudo chown -R ${run_as_user}:${run_as_group} ${sisense_dir} ${verbose}"
    fi
  fi
  log_green "All application directories are valid"
}

# Not using this ugly function, but keeping it
# Just incase we hit a use case which needs it
function ensure_k8s_nodes_etc_hosts() {
  log "Ensuring /etc/hosts containes all k8s nodes"
  local cluster_name=rke-cluster
  local node_count=0
  node_count=$(set | grep '^k8s_nodes_.*_internal_ip=' | wc -l)

  for ((i=1; i<=node_count; i++)); do
    local ip_addr=""
    local node_name=""
    local node_var_name=""
    local ip_var_name=""
    node_var_name=$(set | grep 'k8s_nodes_.*_internal_ip=' | grep _${i}_ | cut -d= -f1 | sed 's/__internal_ip//' | sed 's/_internal_ip//')
    ip_var_name=$(set | grep ${node_var_name} | grep internal_ip= | cut -d= -f1)
    if check_condition ${remote_installation}; then
      ip_var_name=$(set | grep ${node_var_name} | grep external_ip= | cut -d= -f1)
    fi
    ip_addr="${!ip_var_name}"
    node_name=$(set | grep ^${node_var_name}= | cut -d= -f2 | sed 's/node://g' | sed 's/ //g' | sed "s/'//g")
    if [[ -z ${node_name} ]]; then
      # In multi node cluster, the variable could be "k8s_nodes_1_node", "k8s_nodes_2_node", etc...
      export node_name=$(set | grep ^${node_var_name}_node= | cut -d= -f2 | sed 's/node://g' | sed 's/ //g' | sed "s/'//g")
    fi

    if ! sudo grep ${ip_addr} /etc/hosts | grep ${node_name} | grep "${node_name}.${cluster_name}" > /dev/null; then
      # Example:
      # 10.176.30.124 node2.rke-cluster node2 >> /etc/hosts
      log "Adding ${ip_addr} ${node_name}.${cluster_name} ${node_name} into /etc/hosts"
      echo "${ip_addr} ${node_name}.${cluster_name} ${node_name}" | sudo tee -a /etc/hosts > /dev/null
      handle_exit_status "adding ${ip_addr} ${node_name}.${cluster_name} ${node_name} into /etc/hosts"
    else
      log "File /etc/hosts already contains entry ${ip_addr} ${node_name}.${cluster_name} ${node_name}"
    fi
  done

  log_green "File /etc/hosts is valid"
}

function install_storage_utils() {
  if ! check_condition ${offline_installer}; then
    local package=""
    local pkg=""
    pkg=$(which dnf || which yum || which apt)
    if [[ -z ${pkg} ]]; then
      log_stderr "Error: No package manager found!" 1
      handle_exit_status "storage utils installation"
    fi

    if [[ ${storage_type,,} == "rook-ceph" ]]; then
      package=lvm2
    elif [[ ${storage_type,,} == "nfs" || ${storage_type,,} == "efs" ]]; then
      if [[ ${OS} == "Ubuntu" ]]; then
        package=nfs-common
      else
        package=nfs-utils
      fi
    fi

    if [[ ${OS} == "Ubuntu" ]]; then
      update_cmd="sudo ${pkg} update &&"
    fi

    if [[ -n ${package} ]]; then
      run_command "${update_cmd} sudo ${pkg} install -y ${package}" "Installing ${package} on ${OS}"
      log_green "Storage utils ${package} installed successfully"
    fi
  fi
}

function set_hostname() {
  if [[ -n ${NODE_NAME} ]]; then
    run_command "sudo hostnamectl set-hostname ${NODE_NAME}" "Setting hostname to ${NODE_NAME}"
  fi
}


################
##### MAIN #####
################
set_node_name_and_ip
detect_os
if ! check_condition ${offline_installer}; then
  # In offline - customer must install and configure Docker by himself
  # It's mentioned in Sisense docs as mandatory.
  release_dpkg_lock
  bootstrap_os
fi
disable_swap
disable_seliux
enable_kernel_modules "$DEFAULT_KERNEL_MODULES"
enable_kernel_modules "$IPVS_KERNEL_MODULES"
modify_sysctl_entries
set_user_namespaces_sysctl
configure_kernel_runtime_parameters
create_kubernetes_manifests_folder
ensure_sisense_directories
install_storage_utils
set_hostname

log_green "Infrastructure installation completed successfully on node ${NODE_NAME}"
