#!/usr/bin/env bash

source installer/common/functions.sh

function validate_user_config() {
  is_user_exists=$(getent passwd "$config_linux_user")
  if [[ -z $config_linux_user || -z $is_user_exists ]]; then
    log_stderr "Failed to find linux user $config_linux_user ... Please set an existing user with sudo permission in the configuration file and try again..."
    ret 2
  else
    log_to_sisense_installer "Linux user: ${config_linux_user:=$config_run_as_user}"
  fi
  handle_exit_status "validate_user_config"
}

function validate_sudo_permissions() {
  if ! check_condition "$config_offline_installer"; then
    log_to_sisense_installer "Validating Sudo permissions for user $config_linux_user ..."
    if bash -c "timeout -k 2 2 sudo /bin/chmod --help >/dev/null 2>&1" >/dev/null 2>&1; then
      log_to_sisense_installer "User $config_linux_user has sufficient sudo permissions"
    else
      log_stderr "User $config_linux_user has insufficient sudo permissions"
      log_stderr "sudo configuration should include NOPASSWD for the installation and can be removed at the end of the installation."
      ret 2
    fi
    handle_exit_status "validate_sudo_permissions"
  fi
}

function detect_os() {
  log_to_sisense_installer "Detecting Host OS  ..."
  if [ -f /etc/os-release ] && [ -f /etc/redhat-release ]; then
    # freedesktop.org and systemd
    . /etc/os-release
    OS=$NAME
    VER=$VERSION_ID
    major_version=$(echo "$VER" | awk -F'.' '{print $1}')
  elif type lsb_release >>sisense-installer.log 2>&1; then
    # linuxbase.org
    OS=$(lsb_release -si)
    VER=$(lsb_release -sr)
  elif [ -f /etc/lsb-release ]; then
    # For some versions of Debian/Ubuntu without lsb_release command
    . /etc/lsb-release
    OS=$DISTRIB_ID
    VER=$DISTRIB_RELEASE
  elif [ -f /etc/debian_version ]; then
    # Older Debian/Ubuntu/etc.
    OS=Debian
    VER=$(cat /etc/debian_version)
  elif [[ -f /etc/system-release && $(cat /etc/system-release) == *"Amazon Linux"* ]]; then
    . /etc/os-release
    OS="Amazon Linux"
    VER=$VERSION_ID
  else
    # Fall back to uname, e.g. "Linux <version>", also works for BSD, etc.
    OS=$(uname -s)
    VER=$(uname -r)
  fi
  handle_exit_status "detect_os"
  log_to_sisense_installer "OS: $OS, Version: $VER"
}

function validate_os_version() {
  if ! check_condition "$config_offline_installer"; then
    log_to_sisense_installer "Validating OS and its version"

    if [[ ${OS} == *"Red Hat"* ]]; then
      if [[ $VER != "8"* ]] && [[ $VER != "9."* ]]; then
        log_stderr "Unsupported version has been detected, Sisense Linux deployment is certified to run on Red Hat 8.x/9.x only"
        ret 2
      else sudo yum install -y gawk >>sisense-installer.log 2>&1; fi

    elif [[ ${OS} == "Ubuntu" ]]; then
      if [[ ! $VER =~ ^(22\.04|23\.04|24\.04)$ ]]; then
        log_stderr "Unsupported version has been detected, Sisense Linux deployment is certified to run on Ubuntu 22.04/23.04/24.04 only"
        ret 2
      else sudo apt install -y gawk >>sisense-installer.log 2>&1; fi

    elif [[ ${OS} == *"CentOS"* ]]; then
      if [[ $VER != "8"* ]] && [[ $VER != "9"* ]]; then
        log_stderr "Unsupported version has been detected, Sisense Linux deployment is certified to run on CentOS 8.x/9.x only "
        ret 2
      else sudo yum install -y gawk >>sisense-installer.log 2>&1; fi
    elif [[ ${OS} == *"Rocky"* ]]; then
      sudo yum install -y gawk >>sisense-installer.log 2>&1

    elif [[ ${OS} == *"Amazon"* ]]; then
      if [[ $VER != "2" && $VER != "2023" ]]; then
        log_stderr "Unsupported version has been detected, Sisense Linux deployment is certified to run on Amazon Linux 2 and 2023 only"
        ret 2
      else sudo yum install -y gawk >>sisense-installer.log 2>&1; fi
    else
      log_stderr "OS is not supported, please vist https://documentation.sisense.com/latest/linux/step1.htm for more information"
      ret 2
    fi
    handle_exit_status "validate os version"
  fi
}

function validate_no_old_rke() {
  if need_infra_installation || check_condition ${config_uninstall_cluster}; then
    log_to_sisense_installer "Validating old RKE1 is not running here ..."

    local is_old_rke=$(kubectl get pods -n kube-system 2>/dev/null | grep rke- | wc -l)
    if [[ ${is_old_rke} -gt 0 ]]; then
      if ! check_condition ${config_uninstall_cluster}; then
        local msg="upgrade k8s from old RKE1 to new RKE2"
        log_stderr "Error: Cannot ${msg}."
        log_stderr "If your wish to ${msg}, please follow Sisense documentation guidelines." 1
      else
        export old_rke=true
        log_royal_blue "Continuing with old RKE1 uninstallation"
      fi
    else
      log_to_sisense_installer "Can proceed with RKE2 process ..."
    fi
    handle_exit_status "old RKE1 validation"
  fi
}

function validate_namespace_name() {
  log_to_sisense_installer "Validating that namespace name is all lower case ..."
  if [[ "$config_namespace_name" =~ [[:upper:]] ]]; then
    log_to_sisense_installer "Namespace with uppercase detected $config_namespace_name converting it to lower case ..."
    config_namespace_name=$(echo "$config_namespace_name" | tr '[:upper:]' '[:lower:]')
    sed -i "s/namespace_name:.*/namespace_name: $config_namespace_name/g" $VARS_FILE
  fi
  handle_exit_status "validate_namespace_name"
}

function test_ssh_connection() {
  # $1 = pem key
  # $2 = ip address
  echo -n "Validating ssh authentication to $2"
  if [[ -z "$1" ]]; then
    sshpass -p "$config_password" ssh -o "StrictHostKeyChecking no" -q "$config_linux_user@$2" exit
  else
    ssh -o "StrictHostKeyChecking no" -q -i "$1" "$config_linux_user@$2" exit
  fi
  result=$?
  if [[ $result == 0 ]]; then
    echo -e " ...Authentication OK"
    ret 0
  else
    echo -e " ...Fail to authenticate $config_linux_user@$2"
    ret 2
  fi
}

function validate_network_connectivity() {
  if is_on_prem && [[ -z "$config_rwx_sc_name" ]] && [[ -z "$config_rwo_sc_name" ]]; then

    [[ ${config_remote_installation} == "false" ]] && ip_type="internal_ip" || ip_type="external_ip"

    ssh_con_result=0
    trap '(( ssh_con_result |= $? ))' ERR
    echo
    echo -e "Validating ssh connectivity ..."
    for ip in $(yq . $VARS_FILE | jq .k8s_nodes[].$ip_type); do
      target_ip=$(echo "$ip" | tr -d '"')
      nc -zv -w 10 "$target_ip" 22 >/dev/null 2>&1
      result=$?
      echo -n "Validating ssh connection to $target_ip"
      if [[ $result == 0 ]]; then
        echo -e " ...Connection OK"
      else
        echo -e " ...Fail to connect $target_ip:22"
      fi
    done
    trap - ERR

    [[ $ssh_con_result != 0 ]] && echo -e "ERROR - Fail to connect, Please make sure port 22 is open and the ip addresses are correct"
    echo
    echo -e "Validating ssh authentication ..."
    ssh_auth_result=0
    trap '(( ssh_auth_result |= $? ))' ERR
    for ip in $(yq . $VARS_FILE | jq .k8s_nodes[].$ip_type); do
      target_ip=$(echo "$ip" | tr -d '"')
      test_ssh_connection "$config_ssh_key" "$target_ip"
    done
    trap - ERR
    echo
    [[ $ssh_auth_result != 0 ]] && echo -e "ERROR - Fail to open SSH connection with authenticate user $config_linux_user, Please make sure user and password or ssh_key are correct"
    echo
    ret $((ssh_con_result + ssh_auth_result))
    handle_exit_status "ssh concectivity and authentication tests"
  fi
}

function check_k8s_array() {
  if [ -z "$config_k8s_nodes_" ]; then
    log_stderr "Error: k8s_nodes array is empty or currupted - make sure the k8s nodes are correctly defined and indented in the config file (2 spaces)"
    ret 2
    handle_exit_status "check_k8s_array"
  fi
}

function release_dpkg_lock() {
  #TODO - maybe change function name
  if ! check_condition "$config_offline_installer"; then
    if [[ ${OS} == "Ubuntu" ]] || [[ ${OS} == "Debian" ]]; then
      if [ -f "/var/lib/dpkg/lock" ]; then
        log_to_sisense_installer "detected lock on dpkg, releasing it"
        sudo rm -f /var/lib/dpkg/lock
        handle_exit_status "release_dpkg_lock"
      fi
    elif [[ ${OS} == *"Red Hat"* ]] || [[ ${OS} == *"CentOS"* ]] || [[ ${OS} == *"Rocky"* ]]; then
      if [ -f /var/run/yum.pid ]; then
        log_to_sisense_installer "detected lock on yum, releasing it"
        sudo rm -rf /var/run/yum.pid
        handle_exit_status "release_dpkg_lock"
      fi
    fi
  fi
}

function get_sshpass() {
  ASK_PASS=""
  if [[ -n "$config_password" ]]; then
    log_to_sisense_installer "Using password for Sisense Installation ..."
    ASK_PASS="ansible_password=$config_password ansible_sudo_pass=$config_password"
  elif [[ -z "$config_ssh_key" ]]; then
    echo -n "Enter the password for user $config_linux_user and press [ENTER]: "
    read -rs config_password
    echo -e "\nUsing password for Sisense Installation ..."
    ASK_PASS="ansible_password=$config_password ansible_sudo_pass=$config_password"
  elif [[ -n "$config_ssh_key" ]]; then
    config_ssh_key="${config_ssh_key/#\~/$HOME}"
    if [[ ! -f "$config_ssh_key" ]]; then
      log_stderr "SSH key path, $config_ssh_key is not a valid path"
      ret 2
    else
      log_to_sisense_installer "Using private key path $config_ssh_key for Sisense Installation..."
      chmod 400 "$config_ssh_key"
      ASK_PASS="ansible_ssh_private_key_file=$config_ssh_key"
    fi
  else
    ret 2
  fi
  handle_exit_status "get_sshpass"
}

function get_registry_credentials() {
  if [[ -z ${config_docker_username} || -z ${config_docker_password} ]] && not_uninstall && check_condition ${config_private_docker_registry}; then
    echo -n "Enter the username with permissions to pull images from $config_docker_registry and press [ENTER]: "
    read -r docker_username
    export docker_username
    export config_docker_username=${docker_username}

    echo -n "Enter your password and press [ENTER]: "
    read -rs docker_password
    export docker_password
    export config_docker_password=${docker_password}
  fi
  export pull_secrets_name=${config_pull_secrets_name:-sisense-pull-secret}
  handle_exit_status "get_registry_credentials"
}

function validate_k8s_configuration_syntax() {
  node_name_validation=0
  node_name_lowercase_validation=0
  roles_validation=0
  internal_ip_validation=0
  external_ip_validation=0
  disk_volume_validation=0
  overall_validation_verdict=0
  for i in $(eval echo "{0..$(($(yq . $VARS_FILE | jq -r '.k8s_nodes[].node' | wc -l) - 1))}"); do

    [[ ! $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].node') =~ [_] ]]
    node_name_validation=$?
    [[ ! $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].node') =~ [[:upper:]] ]]
    node_name_lowercase_validation=$?
    [[ $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].roles') =~ (^build$)|(^application\, query$)|(^query\, build$)|(^application\, query\, build$)|(^application$)|(^query$) ]]
    roles_validation=$?
    [[ $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].internal_ip') =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]] &&
      [[ ! $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].internal_ip') =~ 0\.0\.0\.0 ]] &&
      [[ ! $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].internal_ip') =~ 127\.0\.0\.0 ]]
    internal_ip_validation=$?
    [[ $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].external_ip') =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]] &&
      [[ ! $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].external_ip') =~ 0\.0\.0\.0 ]] &&
      [[ ! $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].external_ip') =~ 127\.0\.0\.1 ]]
    external_ip_validation=$?
    [[ $(yq . $VARS_FILE | jq -r '.k8s_nodes['"$i"'].disk_volume_device') =~ /dev/.* ]]
    disk_volume_validation=$?
    # skip validation tests for cloud, single and existing cluster configuration
    if check_condition "$config_is_kubernetes_cloud" || check_condition "$config_is_openshift" || check_condition "$config_offline_installer" ||
      [[ -n "$config_rwx_sc_name" ]] || [[ -n "$config_rwo_sc_name" ]]; then
      internal_ip_validation=0
      external_ip_validation=0
      disk_volume_validation=0
    fi
    if ! check_condition "$cluster_mode" || check_condition "$config_offline_installer"; then
      roles_validation=0
      disk_volume_validation=0
    fi

    validation_verdict=$((node_name_validation + node_name_lowercase_validation + internal_ip_validation + external_ip_validation + disk_volume_validation + roles_validation))
    if [[ $validation_verdict != 0 ]]; then
      echo -e "\nWrong syntax was found at : $(yq . $VARS_FILE | jq -r -c '.k8s_nodes['"$i"']')\nPlease verify the following:"
      [[ $node_name_validation != 0 ]] && echo -e "- node: \"$(yq . $VARS_FILE | jq -r -c '.k8s_nodes['"$i"'].node')\" must not contain '_'"
      [[ $node_name_lowercase_validation != 0 ]] && echo -e "- node: \"$(yq . $VARS_FILE | jq -r -c '.k8s_nodes['"$i"'].node')\" must contain lowercase letters only"
      [[ $internal_ip_validation != 0 ]] && echo -e "- internal_ip: \"$(yq . $VARS_FILE | jq -r -c '.k8s_nodes['"$i"'].internal_ip')\" must follow IPv4 structure x.x.x.x or be different from 127.0.0.0 and 0.0.0.0"
      [[ $external_ip_validation != 0 ]] && echo -e "- external_ip: \"$(yq . $VARS_FILE | jq -r -c '.k8s_nodes['"$i"'].external_ip')\" must follow IPv4 structure x.x.x.x or be different from 127.0.0.0 and 0.0.0.0"
      [[ $disk_volume_validation != 0 ]] && echo -e "- disk_volume_device: \"$(yq . $VARS_FILE | jq -r -c '.k8s_nodes['"$i"'].disk_volume_device')\" must follow /dev/xxxx syntax"
      [[ $roles_validation != 0 ]] && echo -e "- roles: \"$(yq . $VARS_FILE | jq -r -c '.k8s_nodes['"$i"'].roles')\" possible values are \"applicationֿ\", \"query\", \"build\" , \"query, build\" or \"application, query, build\""
      ((overall_validation_verdict += validation_verdict))
    fi
  done
  echo
  ret "$overall_validation_verdict"
  handle_exit_status "validate k8s configuration syntax"
}

function validate_k8s_version_on_update() {
  if is_on_prem && not_uninstall && ! check_condition "$config_offline_installer" && check_condition "$config_update" && ! check_condition "${update_k8s_version}"; then
    if ! kubectl version >/dev/null; then
      log_stderr "Error: connection to kubernetes server failed - please verify cluster with sisense intallation exists and accessible." 2
      handle_exit_status "validate_k8s_version_on_update"
    fi
    current_kubernetes_minor_version=$(kubectl version -o yaml | yq -r .serverVersion.minor | sed 's/[^0-9]*//g')
    new_kubernetes_minor_version=22
    if ((current_kubernetes_minor_version < new_kubernetes_minor_version)); then
      log_stderr "Error: Current Kubernetes minor version is: $current_kubernetes_minor_version"
      log_stderr "This version is earlier than the current version: $new_kubernetes_minor_version"
      log_stderr "Updating Sisense from $current_kubernetes_minor_version to $new_kubernetes_minor_version Kubernetes minor version is not supported"
      log_stderr "To upgrade Kubernetes to the supported version, go to the following link: https://documentation.sisense.com/latest/linux/upgrade_k8s.htm#gsc.tab=0"
      ret 2
      handle_exit_status "validate_k8s_version_on_update"
    fi
  fi
}

function validate_common_config_toggles_syntax() {
  for toggle in internal_monitoring external_monitoring uninstall_cluster uninstall_sisense remove_user_data; do
    if [[ $(grep $toggle $VARS_FILE) ]] && [[ ! $(yq . $VARS_FILE | jq -r --arg toggle "$toggle" '.[$toggle]') =~ (false)|(true) ]]; then
      echo "\"$toggle\" toggle value can be true or false, got \"$(yq . $VARS_FILE | jq -r --arg toggle "$toggle" '.[$toggle]')\"."
      ret 2
      handle_exit_status "validate_common_config_toggles_syntax"
    fi
  done
  echo
}

function validate_general_configuration_syntax() {
  #TODO - maybe change this function name

  COLOR_OFF='\033[0m'
  RED='\033[0;31m'
  GREEN='\033[0;32m'

  docker_registry_validation=1

  [[ $(yq . $VARS_FILE | jq -r '.docker_registry') == '' ]]
  docker_registry_validation=$?

  if [[ $docker_registry_validation == 0 ]]; then
    echo -e "${RED}** Docker Registry enabled but not defined.. Exiting. **"
    echo ""
    echo -e "${COLOR_OFF}"
    exit
  fi

  echo
  handle_exit_status "validate General configuration syntax"
}

function validate_outbound_connectivity() {
  if ! check_condition "$config_offline_installer" && not_uninstall; then
    out_con_result=0
    trap '(( out_con_result |= $? ))' ERR
    #todo - cleanup unused sites
    common_sites=("docker.io" "pypi.org" "github.com" "auth.cloud.sisense.com" "bitbucket.org"
      "download.docker.com" "github.com" "gcr.io" "kubernetes.io" "l.sisense.com" "ppa.launchpad.net"
      "quay.io" "registry-1.docker.io" "storage.googleapis.com")
    ubuntu_sites=("ubuntu.com")
    yum_sites=("dl.fedoraproject.org")
    yum_http_sites=("mirror.centos.org")
    case "$OS" in
      Ubuntu)
        sites=("${ubuntu_sites[@]}" "${common_sites[@]}")
        ;;
      *)
        sites=("${yum_sites[@]}" "${common_sites[@]}")
        http_sites=("${yum_http_sites[@]}")
        ;;
    esac
    if [[ -n $config_http_proxy || -n $config_https_proxy ]]; then
      if [[ ! -n $config_no_proxy ]]; then
        log_stderr "no_proxy is not configured, this can cause issues."
      fi 

      export http_proxy=$config_http_proxy
      export https_proxy=$config_https_proxy
      export no_proxy=$config_no_proxy

      if [[ $(nc -h 2>&1 | grep OpenBSD -c) -gt 0 ]]; then
        PROXY_FLAGS="-x${config_https_proxy#*//} -Xconnect"
      else
        PROXY_FLAGS="--proxy ${config_https_proxy#*//} --proxy-type http"
      fi
      
      for site in "${sites[@]}"; do
        nc -zv -w 10 ${PROXY_FLAGS} "$site" 443 >/dev/null 2>&1
        result=$?
        echo -n "Validating connection to $site using https proxy "
        if [[ $result == 0 ]]; then
          echo -e " ... [OK]"
        else
          echo -e " ... [Failed]"
        fi
      done
      if [[ -n "$http_sites" ]]; then
        for http_site in "${http_sites[@]}"; do
          nc -zv -w 10 ${PROXY_FLAGS} "$site" 443 >/dev/null 2>&1
          result=$?
          echo -n "Validating connection to $http_site using http proxy"
          if [[ $result == 0 ]]; then
            echo -e " ... [OK]"
          else
            echo -e " ... [Failed]"
          fi
        done
      fi
    else
      for site in "${sites[@]}"; do
        # retry if netcat fails
        local max_retry=50
        for n in $(seq $max_retry); do
          nc -zv -w 10 "$site" 443 >/dev/null 2>&1
          result=$?
          echo -n "Validating connection to $site on port 443"
          if [[ $result == 0 ]]; then
            echo -e " ... [OK]"
            break
          else
            echo -e " ... [Failed, trying again..${n}]"
            sleep 2
            out_con_result=0
          fi
        done
      done
      if [[ -n "$http_sites" ]]; then
        for http_site in "${http_sites[@]}"; do
          # retry if netcat fails
          local max_retry=50
          for n in $(seq $max_retry); do
            nc -zv -w 10 "$http_site" 80 >/dev/null 2>&1
            result=$?
            echo -n "Validating connection to $http_site on port 80"
            if [[ $result == 0 ]]; then
              echo -e " ... [OK]"
              break
            else
              echo -e " ... [Failed, trying again..${n}]"
              sleep 2
              out_con_result=0
            fi
          done
        done
      fi
    fi
    trap - ERR
    [[ $out_con_result != 0 ]] && echo "ERROR - outband validation failed for one or more sites ... Please make sure it not being blocked by firewall or proxy settings"
    ret $out_con_result
    handle_exit_status "validate outbound connectivity"
  fi
}

function validate_private_registry_connectivity() {
  #TODO - remove this function in the close future
  private_registry_url="quay.io/sisense"
  if [[ "$config_docker_registry" == "$private_registry_url" ]]; then
    echo -n "Validating connection to $private_registry_url"
    if nc -zv -w 10 "$private_registry_url" 80 >/dev/null 2>&1; then
      echo -e " ... [OK]"
    else
      echo -e " ... [Failed]"
      ret 2
    fi
  fi
  handle_exit_status "validate_private_registry_connectivity"
}

function validate_ssl_certs() {
  if check_condition "$config_is_ssl" && ! check_condition "$config_cloud_load_balancer"; then
    if [ ! -f "$config_ssl_cer_path" ] || [ ! -f "$config_ssl_key_path" ]; then
      log_stderr "Error: SSL certificate or key file does not exist"
      ret 2
      handle_exit_status "validate_ssl_certs"
    fi
  fi
}

function validate_storage_type() {
  if $cluster_mode; then
    is_storage_type_filled=$(yq '.storage_type | length > 0' "$VARS_FILE")
    if ! $is_storage_type_filled; then
      log_stderr "You must provide valid storage_type value for cluster mode installations, as mentioned in config file." 2
      handle_exit_status "validate_storage_type"
    fi
  fi
}

function validate_first_node_installer() {
  COLOR_OFF='\033[0m'
  RED='\033[0;31m'
  GREEN='\033[0;32m'

  if ! [ -x "$(command -v ip)" ]; then
    echo -e "${RED}** Sisense Installer cannot verify Host IP. **"
    echo -e "${RED}** Please make sure installing from the first k8s node **"
    echo -e "${COLOR_OFF}"
  else
    CheckIP=$(ip -4 -o addr show scope global | awk '{gsub(/\/.*/,"",$4); print $4}' | grep $(yq . $VARS_FILE | jq -r -c '.k8s_nodes['0'].internal_ip'))

    if is_on_prem && ! check_condition "$config_remote_installation" && ! check_condition "$config_offline_installer" &&
      [[ -z "$config_rwx_sc_name" ]] && [[ -z "$config_rwo_sc_name" ]]; then
      if [ -z "$CheckIP" ]; then

        if [ "$cluster_mode" == "true" ]; then
          echo -e "${RED}** Sisense Installer must run from the first k8s node of the cluster **"
        else
          echo -e "${RED}** Sisense Installer local IP address for k8s node does not match **"
        fi
        echo -e "${RED}** Please check k8s_nodes IP address in config file **"
        echo -e "${RED}** Installer will exit... **"
        echo -e ""
        echo -e "${COLOR_OFF}"
        handle_exit_status "validate_first_node_installer"
        exit 1
      else
        echo -e "${GREEN}Sisense Installer running IP Address for k8s node has been verified."
      fi
    fi

  fi
  echo -e "${COLOR_OFF}"
}

function check_if_sisense_cluster() {
  #if is_on_prem && check_condition "$config_offline_installer"; then
  if is_on_prem; then
    if check_condition "$config_uninstall_cluster"; then
      sisense_cluster="false"
      if kubectl api-resources >/dev/null 2>&1; then
        NODES_JSON=$(kubectl get nodes -o json)
        if jq -e '.items[].metadata.labels."sisense-cluster"' <<<"$NODES_JSON" >/dev/null 2>&1; then
          sisense_cluster="true"
        fi
      # Enforce cluster uninstall if there is no connection to cluster
      else
        sisense_cluster="true"
      fi
      handle_exit_status "check_if_sisense_cluster"
    fi
  fi
}
#todo add traps and error handelling
function generic_pre_install() {
  if ! check_condition "$config_offline_installer"; then
    log_to_sisense_installer "Verifying Python packages exist ..."

    if [[ ${OS} == "Ubuntu" ]]; then
      {
        sudo apt-get update
        sudo DEBIAN_FRONTEND=falseninteractive apt-get install -y python3 netcat-openbsd python3-apt python3-pip dbus pssh sshpass
        sudo rm -f /usr/lib/python$(python3 --version | awk '{print $2}' | awk -F '.' '{print $1"."$2}')/EXTERNALLY-MANAGED 2>&1
        sudo python3 -m pip install --upgrade --force-reinstall pip==21.1.3
        sudo ln -sf /usr/local/bin/pip /usr/bin/pip
        sudo apt-get install -y jq
      } >>sisense-installer.log 2>&1
    elif [[ ${OS} == *"Amazon"* ]]; then
      if [[ "$VER" == "2" ]]; then
        {
          sudo amazon-linux-extras install epel -y
          sudo amazon-linux-extras install python3.8 -y
          #sudo yum reinstall -y python3 python3-pip || sudo yum install -y python3 python3-pip
          sudo ln -sf $(which python3.8) /usr/bin/python3
          sudo yum install -y nc jq libselinux-python3 pssh sshpass
          sudo python3 -m pip install --upgrade --force-reinstall pip==21.1.3
          sudo python3 -m pip install selinux
          sudo ln -sf /usr/local/bin/pip /usr/bin/pip
        } >>sisense-installer.log 2>&1
      elif [[ "$VER" == "2023" ]]; then
        {
          sudo dnf install -y python3 python3-pip python-pip nc jq libselinux-python3 git
          sudo dnf install -y sshpass
          sudo python3 -m pip install --upgrade --force-reinstall pip==21.1.3
          sudo python3 -m pip install selinux
          # Yes, as of 13/08/2023, the only way to install pssh, which are wokring on AL2023, are below commands + install git above...
          sudo python3 -m pip install git+https://github.com/lilydjwg/pssh
          sudo ln -sf /usr/bin/pip /usr/local/bin/pip
        } >>sisense-installer.log 2>&1
      fi
    elif [[ ${OS} == *"CentOS"* ]] || [[ ${OS} == *"Red Hat"* ]] || [[ ${OS} == *"Rocky"* ]]; then
      if [[ ${OS} == *"CentOS"* ]]; then
        pkg_mngr=yum
        epel_version=${VER}
      elif [[ ${OS} == *"Rocky"* ]]; then
        pkg_mngr=yum
        epel_version=${major_version}
      elif [[ ${OS} == *"Red Hat"* ]]; then
        pkg_mngr=dnf
        epel_version=${major_version}
      fi

      {
        sudo ${pkg_mngr} install -y https://dl.fedoraproject.org/pub/epel/epel-release-latest-${epel_version}.noarch.rpm
        sudo ${pkg_mngr} install -y --enablerepo="epel" python3.9 python3-pip nc jq libselinux-python3 pssh sshpass
        sudo ln -sf $(which python3.9) /usr/bin/python3
        sudo python3 -m pip install --upgrade --force-reinstall pip==21.1.3
        sudo ln -sf /usr/local/bin/pip /usr/bin/pip
        sudo python3 -m pip install configparser zipp
      } >>sisense-installer.log 2>&1
    fi

    if [[ ! -d sisense-installer ]]; then
      {
        sudo python3 -m pip install virtualenv
        python3 -m virtualenv sisense-installer
      } >>sisense-installer.log 2>&1
    fi

    source sisense-installer/bin/activate
    log_to_sisense_installer "Installing required pip libraries"
    python3 -m pip install -r installer/requirements.txt --ignore-installed >>sisense-installer.log 2>&1
    handle_exit_status "Installing required pip libraries"
  fi

  if [[ -f /usr/lib/python3/dist-packages/psshlib/manager.py ]]; then
    # Workaround just to remove some warning messages from the log, make the log look nicer...
    sudo sed -i 's/buffering=1/buffering=0/g' /usr/lib/python3/dist-packages/psshlib/manager.py
  fi
}

function ensure_sisense_directory() {
  log_to_sisense_installer "Ensuring sisense main directory ${sisense_dir} exist"
  config_run_as_user=${config_run_as_user:-1000}
  config_run_as_group=${config_run_as_group:-1000}
  if check_condition ${config_update}; then
      pipe_true="|| true"
  fi

  run_command "sudo mkdir -p ${sisense_dir}"
  run_command "sudo chown ${config_run_as_user}:${config_run_as_group} ${sisense_dir} ${pipe_true}"
  run_command "sudo chmod 0755 ${sisense_dir}"

  if ! check_condition ${config_skip_chown_sisense}; then
    local verbose=""
    if check_condition ${config_chown_sisense_verbose}; then
      log_to_sisense_installer "Setting verbose to chown command"
      verbose="-v"
    fi
    if check_condition ${config_chown_sisense_dirs_only}; then
      log_to_sisense_installer "Fixing ${sisense_dir} ownership to directories only"
      run_command "sudo find ${sisense_dir} -type d -exec chown ${config_run_as_user}:${config_run_as_group} ${verbose} {} + ${pipe_true}"
    else
      log_to_sisense_installer "Fixing ${sisense_dir} ownership to all files"
      run_command "sudo chown -R ${config_run_as_user}:${config_run_as_group} ${sisense_dir} ${verbose} ${pipe_true}"
    fi
  fi
}

function configure_sshd_service() {
  if ! check_condition ${config_offline_installer} && is_on_prem; then
    local sshd_config_file=/etc/ssh/sshd_config

    set_property_in_file "ClientAliveInterval" "120" "${sshd_config_file}" "sudo"
    set_property_in_file "ClientAliveCountMax" "30" "${sshd_config_file}" "sudo"
    set_property_in_file "TCPKeepAlive" "yes" "${sshd_config_file}" "sudo"

    if [[ ${OS} == *"Ubuntu"* && $VER =~ ^(23\.04|24\.04)$ ]]; then
      run_command "sudo systemctl restart ssh" "Restarting ssh service"
    else
      run_command "sudo systemctl restart sshd" "Restarting sshd service"
    fi
  fi
}
